<?php

namespace App\Controllers\admin;

use Config\ApiResponseAndNotificationStrings;
use App\Models\Blog_category_model;
use App\Models\Blog_model;
use App\Models\Seo_model;
use App\Models\TranslatedBlogCategoryDetailsModel;
use App\Models\TranslatedBlogDetailsModel;

class Blog extends Admin
{
    public $category, $validation, $db, $ionAuth, $creator_id, $service, $blog;
    protected $superadmin;
    protected ApiResponseAndNotificationStrings $trans;
    protected Seo_model $seoModel;
    protected TranslatedBlogCategoryDetailsModel $translatedBlogCategoryModel;
    protected TranslatedBlogDetailsModel $translatedBlogModel;

    public function __construct()
    {
        parent::__construct();
        $this->category = new Blog_category_model();
        $this->blog = new Blog_model();
        $this->translatedBlogCategoryModel = new TranslatedBlogCategoryDetailsModel();
        $this->translatedBlogModel = new TranslatedBlogDetailsModel();
        $this->db = \Config\Database::connect();
        $this->ionAuth = new \IonAuth\Libraries\IonAuth();
        $this->creator_id = $this->userId;
        $this->superadmin = $this->session->get('email');
        $this->trans = new ApiResponseAndNotificationStrings();
        $this->seoModel = new Seo_model();
        helper('ResponceServices');
    }

    public function index()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            setPageInfo($this->data, labels('blog', 'Blog') . ' | ' . labels('admin_panel', 'Admin Panel'), 'blog');
            // Use the same approach as add_service - get categories with translated names for current system language
            $this->data['categories_name'] = get_categories_with_translated_names();
            return view('backend/admin/template', $this->data);
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - index()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function add_blog_view()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $permission = is_permitted($this->creator_id, 'create', 'blog');
            if (!$permission) {
                return NoPermission();
            }
            setPageInfo($this->data, labels('add_blog', 'Add Blog') . ' | ' . labels('admin_panel', 'Admin Panel'), 'add_blog');
            $this->data['data'] = fetch_details('users', ['id' => $this->userId])[0];
            // Get categories with translations for all languages for dropdown
            // This allows JavaScript to switch between languages without AJAX calls
            $allLanguages = fetch_details('languages', [], ['code']);
            $categoriesData = [];

            foreach ($allLanguages as $lang) {
                $categoriesData[$lang['code']] = $this->category->getCategoriesWithTranslations($lang['code']);
            }

            // Set categories for current system language for initial load
            $currentLanguage = get_current_language(); // Use current system language instead of default
            $this->data['categories_name'] = $categoriesData[$currentLanguage] ?? $categoriesData[$this->getDefaultLanguageCode()];

            // Add languages for translation support (following blog categories pattern)
            $languages = fetch_details('languages', [], ['id', 'language', 'is_default', 'code'], "", '0', 'id', 'ACE');
            $this->data['languages'] = $languages;

            return view('backend/admin/template', $this->data);
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - add_service_view()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function add_blog()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }
            if (isset($_POST) && !empty($_POST)) {
                // Validate main blog fields using Blog_model rules
                $rules = $this->blog->getValidationRules();
                $validation_messages = $this->blog->getValidationMessages();

                if (!$this->validate($rules, $validation_messages)) {
                    return $this->response->setJSON([
                        'error' => true,
                        'message' => $this->validator->getErrors(),
                        'csrfName' => csrf_token(),
                        'csrfHash' => csrf_hash()
                    ]);
                }

                // Get translation data from form (following blog categories pattern)
                $titles = $this->request->getPost('title') ?? [];
                $shortDescriptions = $this->request->getPost('short_description') ?? [];
                $descriptions = $this->request->getPost('description') ?? [];

                // Get default language to use for main table
                $defaultLanguage = $this->getDefaultLanguageCode();

                // Get tags data - handle Tagify JSON format and other formats
                $defaultTagsInput = $this->request->getPost('tags[' . $defaultLanguage . ']') ?? [];
                $defaultTags = $this->parseKeywords($defaultTagsInput);

                // Validate that default language has required fields
                $defaultTitle = isset($titles[$defaultLanguage]) ? trim($titles[$defaultLanguage]) : '';
                $defaultDescription = isset($descriptions[$defaultLanguage]) ? trim($descriptions[$defaultLanguage]) : '';

                if (empty($defaultTitle)) {
                    return ErrorResponse(labels("title_in_default_language_is_required", "Title in default language is required"), true, [], [], 200, csrf_token(), csrf_hash());
                }

                if (empty($defaultDescription)) {
                    return ErrorResponse(labels("description_in_default_language_is_required", "Description in default language is required"), true, [], [], 200, csrf_token(), csrf_hash());
                }

                if (empty($defaultTags)) {
                    return ErrorResponse(labels("tags_in_default_language_are_required", "Tags in default language are required"), true, [], [], 200, csrf_token(), csrf_hash());
                }

                // Handle blog image upload
                $image = '';
                $blogImageFile = $this->request->getFile('blog_image_selector');
                if ($blogImageFile && $blogImageFile->isValid()) {
                    if (!is_dir(FCPATH . 'public/uploads/blogs/images/')) {
                        if (!mkdir(FCPATH . 'public/uploads/blogs/images/', 0775, true)) {
                            return ErrorResponse(labels(FAILED_TO_CREATE_FOLDERS, "Failed to create folders"), true, [], [], 200, csrf_token(), csrf_hash());
                        }
                    }
                    $result = upload_file($blogImageFile, 'public/uploads/blogs/images/', 'Failed to create blogs/images folders', 'blogs/images');

                    if ($result['error'] == false) {
                        $image = $result['file_name'];
                    } else {
                        return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
                    }
                }

                // Generate slug using English content if available, otherwise use blog category logic
                $slug = $this->generateBlogSlug($titles, $this->request->getPost('slug'));

                // Prepare blog data for main table (only non-translatable fields)
                // Translatable fields (title, short_description, description, tags) will be stored in translations table only
                $blogData = [
                    'category_id' => $this->request->getPost('category_id'),
                    'image' => $image,
                    'slug' => $slug,
                ];

                // Save blog using Blog_model::createBlog
                $blogResult = $this->blog->createBlog($blogData);
                if (!empty($blogResult['error'])) {
                    $errors = $blogResult['validation_errors'] ?? [];
                    return ErrorResponse($blogResult['message'] . (!empty($errors) ? ': ' . json_encode($errors) : ''), true, [], [], 200, csrf_token(), csrf_hash());
                }
                $blogId = $blogResult['insert_id'];

                // Process and store translations for all languages (following blog categories pattern)
                try {
                    $translations = [];
                    $allTagData = []; // Store all tag data for proper tag processing

                    foreach ($titles as $languageCode => $title) {
                        // Get tags for this language using the correct form field structure
                        $languageTags = $this->request->getPost('tags[' . $languageCode . ']') ?? [];



                        // Parse tags using the existing parseKeywords method to handle Tagify format
                        $tagsString = $this->parseKeywords($languageTags);


                        // Extract individual tag names for translation processing
                        $tagArray = [];
                        if (!empty($languageTags)) {
                            // Handle different input formats for tags
                            if (is_string($languageTags)) {
                                // Check if it's JSON format from Tagify
                                $decoded = json_decode($languageTags, true);
                                if (is_array($decoded)) {
                                    // Handle Tagify JSON format like [{"value":"new"},{"value":"test"}]
                                    foreach ($decoded as $item) {
                                        if (is_array($item) && isset($item['value'])) {
                                            $tagArray[] = trim($item['value']);
                                        }
                                    }
                                } else {
                                    // Handle comma-separated string format
                                    $tagArray = array_filter(array_map('trim', explode(',', $languageTags)));
                                }
                            } elseif (is_array($languageTags)) {
                                // Handle array format from form submission
                                foreach ($languageTags as $item) {
                                    if (is_string($item)) {
                                        // Check if the string item is JSON (Tagify format)
                                        $decoded = json_decode($item, true);
                                        if (is_array($decoded)) {
                                            // Parse Tagify JSON format like [{"value":"new"},{"value":"test"}]
                                            foreach ($decoded as $tagItem) {
                                                if (is_array($tagItem) && isset($tagItem['value'])) {
                                                    $tagArray[] = trim($tagItem['value']);
                                                }
                                            }
                                        } else {
                                            // Simple string - treat as single tag
                                            $tagArray[] = trim($item);
                                        }
                                    } elseif (is_array($item) && isset($item['value'])) {
                                        // Already parsed Tagify object format
                                        $tagArray[] = trim($item['value']);
                                    }
                                }
                            }
                        }



                        // Store tag data for each language for proper processing
                        $allTagData[$languageCode] = $tagArray;

                        $translations[$languageCode] = [
                            'title' => $this->removeScript(trim($title)),
                            'short_description' => $this->removeScript(trim($shortDescriptions[$languageCode] ?? '')),
                            'description' => $this->removeScript(trim($descriptions[$languageCode] ?? '')),
                            'tags' => $tagsString
                        ];
                    }

                    // Save translations using the optimized model method
                    $translationResult = $this->translatedBlogModel->saveTranslationsOptimized($blogId, $translations);

                    if (!$translationResult) {
                        log_message('warning', 'Some blog translations may not have been saved for blog ID: ' . $blogId);
                    }

                    // Process tags with multi-language support
                    // This ensures tags are stored in blog_tags table and their translations in translated_blog_tag_details
                    $this->processTagsWithTranslations($allTagData, $blogId);
                } catch (\Exception $e) {
                    log_message('error', 'Exception in blog translation processing: ' . $e->getMessage());
                    // Don't fail the blog creation if translation processing fails
                    // The blog is already created, so we just log the translation error
                }

                // Save SEO settings for this blog (modular, like Partners)
                $this->saveSeoSettings($blogId);

                return successResponse(labels(DATA_SAVED_SUCCESSFULLY, "Data saved successfully"), false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return redirect()->to('admin/blog');
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - add_blog()');
            return ErrorResponse($th->getMessage() ?: labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function list()
    {
        try {
            $blog_model = new Blog_model();

            // Get current language for translations (similar to blog categories)
            $currentLanguage = isset($_GET['language_code']) && !empty($_GET['language_code'])
                ? esc($_GET['language_code'])
                : get_current_language();

            $params = [
                'limit' => isset($_GET['limit']) && !empty($_GET['limit']) ? (int)$_GET['limit'] : 10,
                'offset' => isset($_GET['offset']) && !empty($_GET['offset']) ? (int)$_GET['offset'] : 0,
                'sort' => isset($_GET['sort']) && !empty($_GET['sort']) ? esc($_GET['sort']) : 'id',
                'order' => isset($_GET['order']) && in_array(strtoupper($_GET['order']), ['ASC', 'DESC']) ? strtoupper($_GET['order']) : 'DESC',
                'search' => isset($_GET['search']) && !empty($_GET['search']) ? esc($_GET['search']) : '',
                'category_id' => isset($_GET['category_filter']) && !empty($_GET['category_filter']) ? (int)$_GET['category_filter'] : '',
                'language_code' => $currentLanguage, // Pass language to model for translations
                'format' => 'datatable',
                'include_operations' => true
            ];

            $data = $blog_model->list($params);
            return $this->response->setJSON($data);
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - list()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function delete_blog()
    {
        try {
            $disk = fetch_current_file_manager();

            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }

            $id = $this->request->getPost('id');
            $blog = $this->blog->find($id);
            if ($blog && !empty($blog['image'])) {
                delete_file_based_on_server('blogs', $blog['image'], $disk);
            }

            // Clean up SEO settings and images before deleting blog
            $this->seoModel->cleanupSeoData($id, 'blogs');

            $deleteResult = $this->blog->delete($id);
            if ($deleteResult) {
                return successResponse(labels(DATA_DELETED_SUCCESSFULLY, "Data deleted successfully"), false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return ErrorResponse(labels(ERROR_OCCURED, "An error occurred"), true, [], [], 200, csrf_token(), csrf_hash());
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - delete_blog()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function edit_blog()
    {
        try {
            helper('function');
            $uri = service('uri');

            if ($this->isLoggedIn && $this->userIsAdmin) {
                $disk = fetch_current_file_manager();

                $blog_id = $uri->getSegments()[3];

                // Initialize models
                $blogModel = new Blog_model();
                $categoryModel = new Blog_category_model();

                // Fetch blog and categories using models
                $blog = $blogModel->getBlogById($blog_id);

                if (!$blog) {
                    throw new \Exception(labels(BLOG_NOT_FOUND, 'Blog not found'));
                }


                $imagePath = 'public/uploads/blogs/images/' . $blog['image'];
                // Handle image path using get_file_url helper function
                $blog['image'] = get_file_url($disk, $imagePath, 'public/backend/assets/default.png', 'blogs');

                // Fetch SEO settings for this blog
                $this->seoModel->setTableContext('blogs');
                $seo_settings = $this->seoModel->getSeoSettingsByReferenceId($blog_id, 'full');

                $this->data['blog_seo_settings'] = $seo_settings;

                $tags = $blogModel->getTagsForBlog($blog_id);
                $this->data['tags'] = $tags;

                // Get all translations for this blog (following blog categories pattern)
                $translations = $this->translatedBlogModel->getAllTranslations($blog_id);

                // Get default language code
                $defaultLanguage = $this->getDefaultLanguageCode();

                // For new implementation: translatable fields should only come from translations table
                // If no translations exist for existing blogs, populate with main table data as fallback
                if (empty($translations)) {
                    // Convert tags array to comma-separated string for default language
                    $tagsString = '';
                    if (!empty($tags)) {
                        $tagNames = array_column($tags, 'name');
                        $tagsString = implode(',', $tagNames);
                    }

                    // For legacy blogs that don't have translations yet, populate with main table data
                    // This ensures the edit form is prefilled with existing blog content
                    $translations = [
                        $defaultLanguage => [
                            'title' => $blog['title'] ?? '',
                            'short_description' => '', // Not available in main table
                            'description' => $blog['description'] ?? '',
                            'tags' => $tagsString // Convert tags array to comma-separated string
                        ]
                    ];
                } else {
                    // If translations exist but some languages are missing, add fallback for missing languages
                    $allLanguages = fetch_details('languages', [], ['code']);
                    foreach ($allLanguages as $lang) {
                        $langCode = $lang['code'];
                        if (!isset($translations[$langCode])) {
                            // For missing language translations, use main table data for default language
                            // and empty strings for other languages
                            if ($langCode === $defaultLanguage) {
                                // Convert tags array to comma-separated string for default language
                                $tagsString = '';
                                if (!empty($tags)) {
                                    $tagNames = array_column($tags, 'name');
                                    $tagsString = implode(',', $tagNames);
                                }

                                $translations[$langCode] = [
                                    'title' => $blog['title'] ?? '',
                                    'short_description' => '',
                                    'description' => $blog['description'] ?? '',
                                    'tags' => $tagsString
                                ];
                            } else {
                                $translations[$langCode] = [
                                    'title' => '',
                                    'short_description' => '',
                                    'description' => '',
                                    'tags' => ''
                                ];
                            }
                        }
                    }
                }

                // Add languages for translation support (following blog categories pattern)
                $languages = fetch_details('languages', [], ['id', 'language', 'is_default', 'code'], "", '0', 'id', 'ACE');
                $this->data['languages'] = $languages;
                $this->data['translations'] = $translations;

                // Set view data
                setPageInfo($this->data, labels('blog', 'Blog') . ' | ' . labels('admin_panel', 'Admin Panel'), 'blog');
                $this->data['blog'] = $blog;
                // Get categories with translations for all languages for dropdown
                $allLanguages = fetch_details('languages', [], ['code']);
                $categoriesData = [];

                foreach ($allLanguages as $lang) {
                    $categoriesData[$lang['code']] = $categoryModel->getCategoriesWithTranslations($lang['code']);
                }

                // Set categories for current system language for initial load
                $currentLanguage = get_current_language(); // Use current system language instead of default
                $defaultLanguage = $this->getDefaultLanguageCode();

                $this->data['categories_name'] = $categoriesData[$currentLanguage] ?? $categoriesData[$defaultLanguage];
                $this->data['main_page'] = 'edit_blog';
                return view('backend/admin/template', $this->data);
            } else {
                return redirect('admin/login');
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - edit_blog()');
            return ErrorResponse($th->getMessage() ?: labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function update_blog()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }

            if ($_POST && !empty($_POST)) {
                $disk = fetch_current_file_manager();

                $rules = $this->blog->getValidationRules();
                $validation_messages = $this->blog->getValidationMessages();

                if (!$this->validate($rules, $validation_messages)) {
                    return $this->response->setJSON([
                        'error' => true,
                        'message' => $this->validator->getErrors(),
                        'csrfName' => csrf_token(),
                        'csrfHash' => csrf_hash()
                    ]);
                }

                $blog_id = $this->request->getPost('blog_id');

                // Get translation data from form (following blog categories pattern)
                $titles = $this->request->getPost('title') ?? [];
                $shortDescriptions = $this->request->getPost('short_description') ?? [];
                $descriptions = $this->request->getPost('description') ?? [];

                // Get default language to use for main table
                $defaultLanguage = $this->getDefaultLanguageCode();

                // Get tags data - handle Tagify JSON format and other formats
                $defaultTagsInput = $this->request->getPost('tags[' . $defaultLanguage . ']') ?? [];
                $defaultTags = $this->parseKeywords($defaultTagsInput);

                // Validate that default language has required fields
                $defaultTitle = isset($titles[$defaultLanguage]) ? trim($titles[$defaultLanguage]) : '';
                $defaultDescription = isset($descriptions[$defaultLanguage]) ? trim($descriptions[$defaultLanguage]) : '';

                if (empty($defaultTitle)) {
                    return ErrorResponse(labels("title_in_default_language_is_required", "Title in default language is required"), true, [], [], 200, csrf_token(), csrf_hash());
                }

                if (empty($defaultDescription)) {
                    return ErrorResponse(labels("description_in_default_language_is_required", "Description in default language is required"), true, [], [], 200, csrf_token(), csrf_hash());
                }

                if (empty($defaultTags)) {
                    return ErrorResponse(labels("tags_in_default_language_are_required", "Tags in default language are required"), true, [], [], 200, csrf_token(), csrf_hash());
                }

                // Generate slug using English content if available, otherwise use blog category logic
                $slug = $this->generateBlogSlug($titles, $this->request->getPost('slug'), $blog_id);

                // Prepare blog data for main table (only non-translatable fields)
                // Translatable fields (title, short_description, description, tags) will be stored in translations table only
                $data = [
                    'category_id' => $this->request->getPost('category_id'),
                    'slug' => $slug,
                ];

                // Prepare files array for model
                $files = [
                    'image' => $this->request->getFile('blog_image_selector'),
                ];

                // Use model for update logic
                $result = $this->blog->updateBlog($blog_id, $data, $files);
                if (!empty($result['error'])) {
                    $errors = $result['validation_errors'] ?? [];
                    return ErrorResponse($result['message'] . (!empty($errors) ? ': ' . json_encode($errors) : ''), true, [], [], 200, csrf_token(), csrf_hash());
                }

                // Process and store translations for all languages (following blog categories pattern)
                try {
                    $translations = [];
                    $allTagData = []; // Store all tag data for proper tag processing

                    foreach ($titles as $languageCode => $title) {
                        // Get tags for this language using the correct form field structure
                        $languageTags = $this->request->getPost('tags[' . $languageCode . ']') ?? [];



                        // Parse tags using the existing parseKeywords method to handle Tagify format
                        $tagsString = $this->parseKeywords($languageTags);


                        // Extract individual tag names for translation processing
                        $tagArray = [];
                        if (!empty($languageTags)) {
                            // Handle different input formats for tags
                            if (is_string($languageTags)) {
                                // Check if it's JSON format from Tagify
                                $decoded = json_decode($languageTags, true);
                                if (is_array($decoded)) {
                                    // Handle Tagify JSON format like [{"value":"new"},{"value":"test"}]
                                    foreach ($decoded as $item) {
                                        if (is_array($item) && isset($item['value'])) {
                                            $tagArray[] = trim($item['value']);
                                        }
                                    }
                                } else {
                                    // Handle comma-separated string format
                                    $tagArray = array_filter(array_map('trim', explode(',', $languageTags)));
                                }
                            } elseif (is_array($languageTags)) {
                                // Handle array format from form submission
                                foreach ($languageTags as $item) {
                                    if (is_string($item)) {
                                        // Check if the string item is JSON (Tagify format)
                                        $decoded = json_decode($item, true);
                                        if (is_array($decoded)) {
                                            // Parse Tagify JSON format like [{"value":"new"},{"value":"test"}]
                                            foreach ($decoded as $tagItem) {
                                                if (is_array($tagItem) && isset($tagItem['value'])) {
                                                    $tagArray[] = trim($tagItem['value']);
                                                }
                                            }
                                        } else {
                                            // Simple string - treat as single tag
                                            $tagArray[] = trim($item);
                                        }
                                    } elseif (is_array($item) && isset($item['value'])) {
                                        // Already parsed Tagify object format
                                        $tagArray[] = trim($item['value']);
                                    }
                                }
                            }
                        }



                        // Store tag data for each language for proper processing
                        $allTagData[$languageCode] = $tagArray;

                        $translations[$languageCode] = [
                            'title' => $this->removeScript(trim($title)),
                            'short_description' => $this->removeScript(trim($shortDescriptions[$languageCode] ?? '')),
                            'description' => $this->removeScript(trim($descriptions[$languageCode] ?? '')),
                            'tags' => $tagsString
                        ];
                    }

                    // Save translations using the optimized model method
                    $translationResult = $this->translatedBlogModel->saveTranslationsOptimized($blog_id, $translations);

                    if (!$translationResult) {
                        log_message('warning', 'Some blog translations may not have been updated for blog ID: ' . $blog_id);
                    }



                    // Process tags with multi-language support for update
                    // This ensures tags are stored in blog_tags table and their translations in translated_blog_tag_details
                    $this->processTagsWithTranslations($allTagData, $blog_id);
                } catch (\Exception $e) {
                    log_message('error', 'Exception in blog translation processing during update: ' . $e->getMessage());
                    // Don't fail the blog update if translation processing fails
                    // The blog is already updated, so we just log the translation error
                }

                // After successfully updating the blog, also update its SEO settings
                $this->saveSeoSettings($blog_id);
                return successResponse(labels(DATA_UPDATED_SUCCESSFULLY, "Data updated successfully"), false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return redirect()->to('admin/blog');
            }
        } catch (\Throwable $th) {
            throw $th;
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - update_blog()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function add_blog_categories_view()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $permission = is_permitted($this->creator_id, 'create', 'blog');
            if (!$permission) {
                return NoPermission();
            }
            setPageInfo($this->data, labels('blog_categories', 'Blog Categories') . ' | ' . labels('admin_panel', 'Admin Panel'), 'add_blog_categories');

            // fetch languages
            $languages = fetch_details('languages', [], ['id', 'language', 'is_default', 'code'], "", '0', 'id', 'ACE');
            $this->data['languages'] = $languages;
            return view('backend/admin/template', $this->data);
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - add_blog_categories_view()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function list_category()
    {
        try {
            $limit = (isset($_GET['limit']) && !empty($_GET['limit'])) ? $_GET['limit'] : 10;
            $offset = (isset($_GET['offset']) && !empty($_GET['offset'])) ? $_GET['offset'] : 0;
            $sort = (isset($_GET['sort']) && !empty($_GET['sort'])) ? $_GET['sort'] : 'id';
            $order = (isset($_GET['order']) && !empty($_GET['order'])) ? $_GET['order'] : 'ASC';
            $search = (isset($_GET['search']) && !empty($_GET['search'])) ? $_GET['search'] : '';
            $where = [];

            // Get current language for translations
            $currentLanguage = get_current_language();

            // Use the simple approach like promocodes - call the model's list method with language
            $data = $this->category->list($search, $limit, $offset, $sort, $order, $where, $currentLanguage);

            return $this->response->setJSON($data);
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . ' --> app/Controllers/admin/Blog.php - list_category()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function add_category()
    {
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            return $this->response->setJSON($result);
        }
        $creator_id = $this->userId;
        $permission = is_permitted($creator_id, 'create', 'blog');
        if (!$permission) {
            return NoPermission();
        }

        try {
            // Get names from POST data (multi-language format)
            $names = $this->request->getPost('name');

            // Get default language to use as the main name
            $defaultLanguage = $this->getDefaultLanguageCode();
            $defaultName = $names[$defaultLanguage] ?? '';

            if (empty(trim($defaultName))) {
                return ErrorResponse(labels("category_name_in_default_language_is_required", "Category name in default language is required"), true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Prepare data for main table (only non-translatable fields)
            $data = [
                'slug' => generate_unique_slug($this->request->getPost('slug'), 'blog_categories')
            ];

            // Create the blog category
            $result = $this->category->createCategory($data);

            if (isset($result['error']) && $result['error'] === false) {
                $categoryId = $result['insert_id'];

                // Process and store translations for all languages
                try {
                    // Save translations using the optimized model method
                    $translationResult = $this->translatedBlogCategoryModel->saveTranslationsOptimized($categoryId, $names);

                    if (!$translationResult) {
                        log_message('warning', 'Some blog category translations may not have been saved for category ID: ' . $categoryId);
                    }
                } catch (\Exception $e) {
                    log_message('error', 'Exception in blog category translation processing: ' . $e->getMessage());
                    // Don't fail the category creation if translation processing fails
                    // The category is already created, so we just log the translation error
                }

                return successResponse($result['message'], false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - add_category()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function update_category()
    {
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            return $this->response->setJSON($result);
        }
        $creatorId = $this->userId;
        $permission = is_permitted($creatorId, 'update', 'blog');
        if (!$permission) {
            return NoPermission();
        }
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }

        try {
            // Get names from POST data (multi-language format)
            $names = $this->request->getPost('name');

            // Get default language to use as the main name
            $defaultLanguage = $this->getDefaultLanguageCode();
            $defaultName = $names[$defaultLanguage] ?? '';

            if (empty(trim($defaultName))) {
                return ErrorResponse(labels("category_name_in_default_language_is_required", "Category name in default language is required"), true, [], [], 200, csrf_token(), csrf_hash());
            }

            $id = $this->request->getPost('id');

            // Prepare data for main table (only non-translatable fields)
            $data = [
                'slug' => $this->request->getPost('slug'),
            ];

            $upd = $this->category->updateCategory($id, $data);

            if (isset($upd['error']) && $upd['error'] == false) {
                // Process and store translations for all languages
                try {
                    // Save translations using the optimized model method
                    $translationResult = $this->translatedBlogCategoryModel->saveTranslationsOptimized($id, $names);

                    if (!$translationResult) {
                        log_message('warning', 'Some blog category translations may not have been updated for category ID: ' . $id);
                    }
                } catch (\Exception $e) {
                    log_message('error', 'Exception in blog category translation processing during update: ' . $e->getMessage());
                    // Don't fail the category update if translation processing fails
                    // The category is already updated, so we just log the translation error
                }

                return successResponse($upd['message'], false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return ErrorResponse($upd['message'], true, [], [], 200, csrf_token(), csrf_hash());
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - update_category()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function remove_category()
    {
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            return $this->response->setJSON($result);
        }
        $creator_id = $this->userId;
        $permission = is_permitted($creator_id, 'delete', 'blog');
        if (!$permission) {
            return NoPermission();
        }
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }

        try {
            $id = $this->request->getPost('id');
            $result = $this->category->deleteCategory($id);

            if (isset($result['error']) && $result['error'] == false) {
                return successResponse($result['message'], false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - remove_category()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    /**
     * Save SEO settings for a blog post, following the pattern from Partners.php
     * @param int $blogId
     */
    private function saveSeoSettings(int $blogId): void
    {
        // Parse meta keywords (Tagify or comma-separated)
        $metaKeywords = $this->request->getPost('meta_keywords');
        $keywords = $this->parseKeywords($metaKeywords);

        // Build SEO data array
        $seoData = [
            'title'         => trim((string) $this->request->getPost('meta_title')),
            'description'   => trim((string) $this->request->getPost('meta_description')),
            'keywords'      => $keywords,
            'schema_markup' => trim((string) $this->request->getPost('schema_markup')),
            'blog_id'       => $blogId,
        ];

        // Check if any SEO field is filled (excluding blog_id)
        $hasSeoData = array_filter($seoData, fn($v) => !empty($v) && $v !== $blogId);

        // Check if all SEO fields are intentionally cleared
        $allFieldsCleared = empty($seoData['title']) &&
            empty($seoData['description']) &&
            empty($seoData['keywords']) &&
            empty($seoData['schema_markup']);

        // Handle SEO image upload
        $seoImage = $this->request->getFile('meta_image');
        $hasImage = $seoImage && $seoImage->isValid();

        // Use Seo_model for blog context
        $this->seoModel->setTableContext('blogs');
        $existingSettings = $this->seoModel->getSeoSettingsByReferenceId($blogId);

        $newSeoData = $seoData;
        if ($hasImage) {
            try {
                if (!is_dir(FCPATH . 'public/uploads/seo_settings/blog_seo_settings/')) {
                    mkdir(FCPATH . 'public/uploads/seo_settings/blog_seo_settings/', 0775, true);
                }
                $uploadResult = upload_file(
                    $seoImage,
                    'public/uploads/seo_settings/blog_seo_settings/',
                    labels(FAILED_TO_UPLOAD_SEO_IMAGE, 'Failed to upload SEO image'),
                    'seo_settings/blog_seo_settings'
                );
                if ($uploadResult['error'] == false) {
                    $newSeoData['image'] = $uploadResult['file_name'];
                } else {
                    throw new \Exception($uploadResult['message']);
                }
            } catch (\Throwable $t) {
                throw new \Exception(labels(SEO_IMAGE_UPLOAD_FAILED, 'SEO image upload failed: ' . $t->getMessage()));
            }
        } else {
            $newSeoData['image'] = $existingSettings['image'] ?? '';
        }

        // If no existing settings, create new if data or image exists
        // This ensures that even if only an image is provided, we create the SEO record
        if (!$existingSettings) {
            if ($hasSeoData || $hasImage) {
                $result = $this->seoModel->createSeoSettings($newSeoData);
                if (!empty($result['error'])) {
                    $errors = $result['validation_errors'] ?? [];
                    throw new \Exception($result['message'] . (!empty($errors) ? ': ' . json_encode($errors) : ''));
                }
            }
            return;
        }

        // If existing settings exist and all fields are cleared (and no new image), delete the record
        // BUT: If there's an existing image, we should NOT delete the record even if all other fields are empty
        // This preserves the SEO record structure for future use
        if ($existingSettings && $allFieldsCleared && !$hasImage && empty($existingSettings['image'])) {
            $result = $this->seoModel->delete($existingSettings['id']);
            if ($result) {
                // Clean up old image if it exists
                if (!empty($existingSettings['image'])) {
                    $disk = fetch_current_file_manager();
                    delete_file_based_on_server('blog_seo_settings', $existingSettings['image'], $disk);
                }
            }
            return;
        }

        // Compare existing and new settings
        $settingsChanged = false;
        foreach ($newSeoData as $key => $value) {
            $existingValue = $existingSettings[$key] ?? '';
            $newValue = $value ?? '';
            if ($existingValue !== $newValue) {
                $settingsChanged = true;
                break;
            }
        }
        if (!$settingsChanged) {
            return;
        }

        // Update existing settings with new data
        $result = $this->seoModel->updateSeoSettings($existingSettings['id'], $newSeoData);
        if (!empty($result['error'])) {
            $errors = $result['validation_errors'] ?? [];
            throw new \Exception($result['message'] . (!empty($errors) ? ': ' . json_encode($errors) : ''));
        }
    }

    /**
     * Remove SEO image for a blog
     * This method handles AJAX requests to remove SEO images
     * @return \CodeIgniter\HTTP\Response
     */
    public function remove_seo_image()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return ErrorResponse(labels(UNAUTHORIZED_ACCESS, "Unauthorized access"), true, [], [], 200, csrf_token(), csrf_hash());
            }

            $blogId = $this->request->getPost('blog_id');
            $seoId = $this->request->getPost('seo_id');

            if (!$blogId) {
                return ErrorResponse(labels(BLOG_ID_REQUIRED, "Blog ID is required"), true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Set SEO model context for blogs
            $this->seoModel->setTableContext('blogs');

            // Get existing SEO settings
            $existingSettings = $this->seoModel->getSeoSettingsByReferenceId($blogId);

            if (!$existingSettings) {
                return ErrorResponse(labels(SEO_SETTINGS_NOT_FOUND_FOR_THIS_BLOG, "SEO settings not found for this blog"), true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Check if there's an image to remove
            if (empty($existingSettings['image'])) {
                return ErrorResponse(labels(NO_SEO_IMAGE_FOUND_TO_REMOVE, "No SEO image found to remove"), true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Store the image name for cleanup
            $imageToDelete = $existingSettings['image'];

            // Prepare update data - remove image but keep other fields
            $updateData = [
                'title' => $existingSettings['title'] ?? '',
                'description' => $existingSettings['description'] ?? '',
                'keywords' => $existingSettings['keywords'] ?? '',
                'schema_markup' => $existingSettings['schema_markup'] ?? '',
                'image' => '', // Clear the image field
                'blog_id' => $blogId
            ];

            // Check if all other SEO fields are empty
            $hasOtherSeoData = !empty($updateData['title']) ||
                !empty($updateData['description']) ||
                !empty($updateData['keywords']) ||
                !empty($updateData['schema_markup']);

            // If all other fields are empty, we should NOT delete the record
            // Instead, we keep the record with empty image but preserve the structure
            // This ensures the SEO record exists for future use
            $result = $this->seoModel->updateSeoSettings($existingSettings['id'], $updateData);

            if (!empty($result['error'])) {
                return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Clean up the image file from storage
            if (!empty($imageToDelete)) {
                $disk = fetch_current_file_manager();
                delete_file_based_on_server('blog_seo_settings', $imageToDelete, $disk);
            }

            return successResponse(labels(SEO_IMAGE_REMOVED_SUCCESSFULLY, "SEO image removed successfully"), false, [], [], 200, csrf_token(), csrf_hash());
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Blog.php - remove_seo_image()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something went wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    /**
     * Parse keywords/tags from Tagify or comma-separated string
     * @param mixed $input
     * @return string
     */
    private function parseKeywords($input): string
    {
        // If input is empty, return empty string
        if (empty($input)) {
            return '';
        }

        // If input is a string, it might be JSON or comma-separated
        if (is_string($input)) {
            // Check if it's a JSON string
            if (json_decode($input, true) !== null) {
                $decoded = json_decode($input, true);
                if (is_array($decoded)) {
                    // Handle array of objects (e.g., [{value: "tag1"}, {value: "tag2"}])
                    $tags = array_map(function ($item) {
                        return is_array($item) && isset($item['value']) ? trim($item['value']) : trim($item);
                    }, $decoded);
                    return implode(',', $tags);
                }
            }
            // Treat as comma-separated string
            return trim($input);
        }

        // If input is an array
        if (is_array($input)) {
            // Handle case where array contains a single JSON string (e.g., ['[{value: "tag1"}, {value: "tag2"}]'])
            if (count($input) === 1 && is_string($input[0]) && json_decode($input[0], true) !== null) {
                $decoded = json_decode($input[0], true);
                if (is_array($decoded)) {
                    $tags = array_map(function ($item) {
                        return is_array($item) && isset($item['value']) ? trim($item['value']) : trim($item);
                    }, $decoded);
                    return implode(',', $tags);
                }
            }
            // Handle array of objects (e.g., [{value: "tag1"}, {value: "tag2"}])
            $tags = array_map(function ($item) {
                return is_array($item) && isset($item['value']) ? trim($item['value']) : trim($item);
            }, $input);
            return implode(',', $tags);
        }

        // Fallback: return empty string for unexpected input
        return '';
    }

    /**
     * Get the default language code from the database
     * 
     * @return string The default language code
     */
    private function getDefaultLanguageCode(): string
    {
        $languages = fetch_details('languages', ['is_default' => 1], ['code']);
        return $languages[0]['code'] ?? 'en';
    }

    /**
     * Generate blog slug with intelligent language prioritization
     * 1. If English content is available, use English title for slug
     * 2. If English is not available, use the same logic as blog categories (posted slug)
     * 
     * @param array $titles Array of titles by language code
     * @param string $postedSlug The slug posted from the form
     * @param int|null $excludeId ID to exclude for updates (optional)
     * @return string Generated unique slug
     */
    private function generateBlogSlug(array $titles, string $postedSlug, int $excludeId = null): string
    {
        // Check if English content is available
        $englishTitle = isset($titles['en']) ? trim($titles['en']) : '';

        if (!empty($englishTitle)) {
            // Use English title for slug generation (prioritize English)
            $baseSlug = url_title($englishTitle, '-', true);
            return generate_unique_slug($baseSlug, 'blogs', $excludeId);
        } else {
            // Fallback to blog category logic: use the posted slug directly
            // This matches the behavior in blog categories where slug is used as-is
            return generate_unique_slug($postedSlug, 'blogs', $excludeId);
        }
    }

    /**
     * Get blog data with translations for editing
     * Follows the same pattern as get_blog_category_data
     * 
     * @return \CodeIgniter\HTTP\Response JSON response with blog data
     */
    public function get_blog_data()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return $this->response->setJSON(['error' => true, 'message' => labels(UNAUTHORIZED_ACCESS)]);
            }

            $blogId = $this->request->getPost('id');
            if (empty($blogId)) {
                return $this->response->setJSON(['error' => true, 'message' => labels(BLOG_ID_REQUIRED)]);
            }

            // Get the blog data
            $blog = $this->blog->find($blogId);
            if (!$blog) {
                return $this->response->setJSON(['error' => true, 'message' => labels(DATA_NOT_FOUND)]);
            }

            // Get all translations for this blog
            $translations = $this->translatedBlogModel->getAllTranslations($blogId);

            // Get default language code
            $defaultLanguage = $this->getDefaultLanguageCode();

            // For new implementation: translatable fields should only come from translations table
            // If no translations exist for existing blogs, this indicates legacy data that needs migration
            if (empty($translations)) {
                // For legacy blogs that don't have translations yet, create empty structure
                // Note: In production, you may want to migrate existing blog data to translations table
                $translations = [
                    $defaultLanguage => [
                        'title' => '',
                        'short_description' => '',
                        'description' => '',
                        'tags' => ''
                    ]
                ];
            }

            // Prepare response data
            $responseData = [
                'id' => $blog['id'],
                'slug' => $blog['slug'],
                'category_id' => $blog['category_id'],
                'image' => $blog['image'],
                'status' => $blog['status'] ?? 1,
                'translations' => $translations
            ];

            return $this->response->setJSON([
                'error' => false,
                'data' => $responseData
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Error getting blog data: ' . $e->getMessage());
            return $this->response->setJSON([
                'error' => true,
                'message' => labels(ERROR_OCCURED)
            ]);
        }
    }

    /**
     * Get categories with translations for a specific language (AJAX endpoint)
     * Used by JavaScript to update category dropdown when language changes
     * 
     * @return \CodeIgniter\HTTP\Response JSON response with categories
     */
    public function get_categories_by_language()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return $this->response->setJSON(['error' => true, 'message' => labels(UNAUTHORIZED_ACCESS)]);
            }

            $languageCode = $this->request->getPost('language_code');
            if (empty($languageCode)) {
                return $this->response->setJSON(['error' => true, 'message' => labels(LANGUAGE_CODE_IS_REQUIRED)]);
            }

            // Get categories with translations for the specified language
            $categories = $this->category->getCategoriesWithTranslations($languageCode);

            return $this->response->setJSON([
                'error' => false,
                'data' => $categories
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Error getting categories by language: ' . $e->getMessage());
            return $this->response->setJSON([
                'error' => true,
                'message' => labels(ERROR_OCCURED)
            ]);
        }
    }

    /**
     * Get blog category data with translations for editing
     * 
     * @return \CodeIgniter\HTTP\Response JSON response with blog category data
     */
    public function get_blog_category_data()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return $this->response->setJSON(['error' => true, 'message' => labels(UNAUTHORIZED_ACCESS)]);
            }

            $categoryId = $this->request->getPost('id');
            if (empty($categoryId)) {
                return $this->response->setJSON(['error' => true, 'message' => labels('category_id_is_required')]);
            }

            // Get the blog category data
            $category = $this->category->find($categoryId);
            if (!$category) {
                return $this->response->setJSON(['error' => true, 'message' => labels(DATA_NOT_FOUND)]);
            }

            // Get all translations for this category
            $translations = $this->translatedBlogCategoryModel->getAllTranslations($categoryId);

            // Get default language code
            $defaultLanguage = $this->getDefaultLanguageCode();

            // If no translations exist, create fallback data using the main table data
            if (empty($translations)) {
                // Create fallback translations object with default language
                // Use the name from main table if it exists, otherwise empty string
                $translations = [
                    $defaultLanguage => $category['name'] ?? ''
                ];
            } else {
                // If translations exist, ensure default language has a value
                // If default language translation is missing, use main table name as fallback
                if (!isset($translations[$defaultLanguage]) || empty($translations[$defaultLanguage])) {
                    $translations[$defaultLanguage] = $category['name'] ?? '';
                }
            }

            // Prepare response data
            $responseData = [
                'id' => $category['id'],
                'slug' => $category['slug'],
                'status' => $category['status'],
                'translations' => $translations
            ];

            return $this->response->setJSON([
                'error' => false,
                'data' => $responseData
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Error getting blog category data: ' . $e->getMessage());
            return $this->response->setJSON([
                'error' => true,
                'message' => labels(ERROR_OCCURED)
            ]);
        }
    }

    /**
     * Process tags with multi-language translations for blog creation
     * This method handles the storage of individual tags and their translations
     * 
     * @param array $allTagData Array of tag data organized by language code
     * @param int $blogId The blog ID to associate tags with
     * @return void
     */
    private function processTagsWithTranslations(array $allTagData, int $blogId): void
    {
        try {
            // Get all unique tag names from all languages to avoid duplicates
            $uniqueTagNames = [];
            $tagTranslations = []; // Store translations for each tag

            // Collect all unique tag names and their translations
            foreach ($allTagData as $languageCode => $tags) {
                if (!is_array($tags)) {
                    continue;
                }

                foreach ($tags as $tagName) {
                    $tagName = trim($tagName);
                    if (empty($tagName)) {
                        continue;
                    }

                    // Store the tag name (use lowercase for uniqueness check)
                    $lowerTagName = strtolower($tagName);
                    if (!isset($uniqueTagNames[$lowerTagName])) {
                        $uniqueTagNames[$lowerTagName] = $tagName; // Keep original case for storage
                        $tagTranslations[$tagName] = []; // Initialize translations array
                    }

                    // Store translation for this tag
                    $tagTranslations[$tagName][$languageCode] = $tagName;
                }
            }

            // If no tags found, return early
            if (empty($uniqueTagNames)) {
                return;
            }

            // Format data according to what processTagsWithTranslations expects
            $formattedTagData = [];
            foreach ($uniqueTagNames as $tagName) {
                $formattedTagData[] = [
                    'name' => $tagName,
                    'translations' => $tagTranslations[$tagName] ?? []
                ];
            }

            // Use the blog model's method to process tags with translations
            $this->blog->processTagsWithTranslations($formattedTagData, $blogId);
        } catch (\Exception $e) {
            log_message('error', 'Error processing tags with translations in Blog controller: ' . $e->getMessage());
            // Don't throw exception to avoid breaking blog creation
        }
    }
}
