<?php

namespace App\Controllers\admin;

use App\Models\Seo_model;
use App\Models\User_permissions_model;

/**
 * SEO Settings Controller
 * Handles all SEO settings related operations in the admin panel
 * Follows CI4 best practices and MVC architecture
 */
class SeoSettings extends Admin
{
    public $creator_id;
    protected $superadmin;
    protected $validation;
    protected $seoModel;
    protected $userPermissions;

    public function __construct()
    {
        parent::__construct();
        $this->creator_id = $this->userId;
        $this->validation = \Config\Services::validation();
        $this->superadmin = $this->session->get('email');
        // Load the SEO model using CI4 best practices
        $this->seoModel = model(Seo_model::class);
        $this->userPermissions = get_permission($this->userId);

        helper('ResponceServices');
    }

    /**
     * Display the SEO settings page
     * 
     * @return \CodeIgniter\HTTP\RedirectResponse|string
     */
    public function index()
    {
        // Check authentication and authorization
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }

        // Set page information for the view
        setPageInfo(
            $this->data,
            labels('seo_settings', 'SEO Settings') . ' | ' . labels('admin_panel', 'Admin Panel'),
            'seo_settings'
        );

        return view('backend/admin/template', $this->data);
    }

    /**
     * Create new SEO settings record
     * Handles form submission and validation using CI4 best practices
     * 
     * @return \CodeIgniter\HTTP\ResponseInterface
     */
    public function add_seo_settings()
    {
        // Check authentication and authorization
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }

        // Check demo mode restrictions
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            return $this->response->setJSON($result);
        }

        try {
            if (!$this->validate($this->seoModel->getValidationRules(), $this->seoModel->getValidationMessages())) {
                return $this->response->setJSON([
                    'error' => true,
                    'message' => labels($this->validator->getErrors()),
                    'csrfName' => csrf_token(),
                    'csrfHash' => csrf_hash()
                ]);
            }

            // Get validated data from form
            $formData = $this->getValidatedData();

            // Handle image upload
            $imageResult = $this->handleImageUpload();
            if ($imageResult['error']) {
                return $this->response->setJSON([
                    'error' => true,
                    'message' => $imageResult['message'],
                    'csrfName' => csrf_token(),
                    'csrfHash' => csrf_hash()
                ]);
            }

            // Prepare data for model
            $seoData = array_merge($formData, [
                'image' => $imageResult['file_name']
            ]);

            // Use model to create the SEO settings
            $result = $this->seoModel->createSeoSettings($seoData);

            return $this->response->setJSON([
                'error' => $result['error'],
                'message' => $result['message'],
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        } catch (\Exception $e) {
            throw $e;
            // Log error for debugging
            log_message('error', 'SEO Settings creation error in controller: ' . $e->getMessage());

            return $this->response->setJSON([
                'error' => true,
                'message' => labels(SOMETHING_WENT_WRONG, 'Something went wrong'),
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        }
    }

    /**
     * Get paginated list of SEO settings
     * Uses the model to fetch data following CI4 best practices
     * 
     * @return \CodeIgniter\HTTP\ResponseInterface
     */
    public function seo_settings_list()
    {
        // Check authentication and authorization
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }

        try {
            // Get parameters from request using CI4 IncomingRequest methods
            $params = [
                'limit' => $this->request->getGet('limit') ?? 10,
                'offset' => $this->request->getGet('offset') ?? 0,
                'sort' => $this->request->getGet('sort') ?? 'id',
                'order' => $this->request->getGet('order') ?? 'DESC',
                'search' => $this->request->getGet('search') ?? ''
            ];

            // Use model to get formatted list data
            $result = $this->seoModel->getFormattedSeoSettingsList($params);

            $permission = [
                'update' => $this->userPermissions['update']['seo_settings'],
                'delete' => $this->userPermissions['delete']['seo_settings']
            ];

            // Add operation buttons to each row
            foreach ($result['rows'] as &$row) {
                $row['operations'] = $this->generateOperationButtons($row['id'], $permission);
            }

            return $this->response->setJSON([
                'total' => $result['total'],
                'rows' => $result['rows']
            ]);
        } catch (\Exception $e) {
            // throw $e;
            log_message('error', 'SEO Settings list error in controller: ' . $e->getMessage());

            return $this->response->setJSON([
                'total' => 0,
                'rows' => [],
                'error' => labels(FAILED_TO_LOAD_SEO_SETTINGS, 'Failed to load SEO settings')
            ]);
        }
    }

    /**
     * Update existing SEO settings record
     * 
     * @return \CodeIgniter\HTTP\ResponseInterface
     */
    public function update_seo_settings()
    {
        // Check authentication and authorization
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }

        // Check demo mode restrictions
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            return $this->response->setJSON($result);
        }

        try {
            // Get and validate the record ID
            $id = $this->request->getPost('id');
            if (!$id || !is_numeric($id)) {
                return $this->response->setJSON([
                    'error' => true,
                    'message' => labels(INVALID_RECORD_ID_PROVIDED, 'Invalid record ID provided'),
                    'csrfName' => csrf_token(),
                    'csrfHash' => csrf_hash()
                ]);
            }

            // Create validation rules for update (ID is required)
            $updateRules = array_merge($this->seoModel->getValidationRules(), [
                'id' => 'required|numeric'
            ]);

            // Use CI4 validation
            if (!$this->validate($updateRules)) {
                return $this->response->setJSON([
                    'error' => true,
                    'message' => $this->validator->getErrors(),
                    'csrfName' => csrf_token(),
                    'csrfHash' => csrf_hash()
                ]);
            }

            // Get validated data from form
            $formData = $this->getValidatedData();

            // Handle image upload if new image is provided
            $image = $this->request->getFile('image');
            if ($image && $image->isValid() && !$image->hasMoved()) {
                // Get existing record to access old image
                $existingRecord = $this->seoModel->getSeoSettingsById($id);
                $oldImageName = $existingRecord ? $existingRecord['image'] : null;

                $imageResult = $this->handleImageUpload();
                if ($imageResult['error']) {
                    return $this->response->setJSON([
                        'error' => true,
                        'message' => $imageResult['message'],
                        'csrfName' => csrf_token(),
                        'csrfHash' => csrf_hash()
                    ]);
                }

                // Set new image name in form data
                $formData['image'] = $imageResult['file_name'];

                // Delete old image file if it exists and is different from new one
                if ($oldImageName && $oldImageName !== $imageResult['file_name']) {
                    $this->cleanupImageFile($oldImageName);
                }
            }

            // Use model to update the SEO settings
            $result = $this->seoModel->updateSeoSettings($id, $formData);

            return $this->response->setJSON([
                'error' => $result['error'],
                'message' => $result['message'],
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        } catch (\Exception $e) {
            log_message('error', 'SEO Settings update error in controller: ' . $e->getMessage());

            return $this->response->setJSON([
                'error' => true,
                'message' => labels(ERROR_OCCURED, 'An error occurred'),
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        }
    }

    /**
     * Delete SEO settings record
     * 
     * @return \CodeIgniter\HTTP\ResponseInterface
     */
    public function delete_seo_settings()
    {
        // Check authentication and authorization
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return redirect('admin/login');
        }

        // Check demo mode restrictions
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            return $this->response->setJSON($result);
        }

        try {
            // Get and validate the record ID
            $id = $this->request->getPost('id');
            if (!$id || !is_numeric($id)) {
                return $this->response->setJSON([
                    'error' => true,
                    'message' => labels(INVALID_RECORD_ID_PROVIDED, 'Invalid record ID provided'),
                    'csrfName' => csrf_token(),
                    'csrfHash' => csrf_hash()
                ]);
            }

            // Use model to delete the SEO settings
            $result = $this->seoModel->deleteSeoSettings($id);

            // If deletion was successful and there's an image, clean it up
            if (!$result['error'] && isset($result['deleted_record']['image'])) {
                $this->cleanupImageFile($result['deleted_record']['image']);
            }

            return $this->response->setJSON([
                'error' => $result['error'],
                'message' => $result['message'],
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        } catch (\Exception $e) {
            log_message('error', 'SEO Settings delete error in controller: ' . $e->getMessage());

            return $this->response->setJSON([
                'error' => true,
                'message' => labels(ERROR_OCCURED, 'An error occurred'),
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        }
    }

    /**
     * Get SEO settings data by ID for editing
     * 
     * @return \CodeIgniter\HTTP\ResponseInterface
     */
    public function get_seo_settings()
    {
        // Check authentication and authorization
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return $this->response->setJSON([
                'error' => true,
                'message' => labels(UNAUTHORIZED_ACCESS, 'Unauthorized access'),
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        }

        try {
            // Get and validate the record ID
            $id = $this->request->getGet('id');
            if (!$id || !is_numeric($id)) {
                return $this->response->setJSON([
                    'error' => true,
                    'message' => labels(INVALID_RECORD_ID_PROVIDED, 'Invalid record ID provided'),
                    'csrfName' => csrf_token(),
                    'csrfHash' => csrf_hash()
                ]);
            }

            // Use model to get the SEO settings
            $seoData = $this->seoModel->getSeoSettingsById($id);

            if (!$seoData) {
                return $this->response->setJSON([
                    'error' => true,
                    'message' => labels(DATA_NOT_FOUND, 'Data not found'),
                    'csrfName' => csrf_token(),
                    'csrfHash' => csrf_hash()
                ]);
            }

            // Add image URL for display using CI4 URL helper
            if (!empty($seoData['image'])) {
                $seoData['image_url'] = $this->seoModel->getImageUrl($seoData['image']);
            }

            return $this->response->setJSON([
                'error' => false,
                'data' => $seoData,
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Get SEO Settings error in controller: ' . $e->getMessage());

            return $this->response->setJSON([
                'error' => true,
                'message' => labels(FAILED_TO_LOAD_DATA, 'Failed to load data'),
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        }
    }

    /**
     * Get existing pages that already have SEO settings
     * Uses the model to fetch data following CI4 best practices
     * 
     * @return \CodeIgniter\HTTP\ResponseInterface
     */
    public function get_existing_pages()
    {
        // Check authentication and authorization
        if (!$this->isLoggedIn || !$this->userIsAdmin) {
            return $this->response->setJSON([
                'error' => true,
                'message' => labels(UNAUTHORIZED_ACCESS, 'Unauthorized access'),
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        }

        try {
            // Use model to get existing pages
            $existingPages = $this->seoModel->getExistingPages();

            return $this->response->setJSON([
                'error' => false,
                'existing_pages' => $existingPages,
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Get existing pages error in controller: ' . $e->getMessage());

            return $this->response->setJSON([
                'error' => true,
                'message' => labels(FAILED_TO_LOAD_DATA, 'Failed to load data'),
                'existing_pages' => [],
                'csrfName' => csrf_token(),
                'csrfHash' => csrf_hash()
            ]);
        }
    }

    // PRIVATE HELPER METHODS FOR CONTROLLER LOGIC

    /**
     * Get validated form data using CI4 validation
     * 
     * @return array - Cleaned and validated form data
     */
    private function getValidatedData()
    {
        // Get keywords array from Tagify
        $keywords = $this->request->getPost('keywords');

        // Convert keywords array to comma-separated string for storage
        // Tagify sends keywords as an array, but we store as comma-separated string
        if (is_array($keywords)) {
            $keywordsString = implode(',', array_map('trim', $keywords));
        } else {
            $keywordsString = is_string($keywords) ? trim($keywords) : '';
        }

        return [
            'page' => $this->request->getPost('page'),
            'title' => $this->request->getPost('title'),
            'description' => $this->request->getPost('description'),
            'keywords' => $keywordsString,
            'schema_markup' => $this->request->getPost('schema_markup')
        ];
    }

    /**
     * Handle image file upload with validation using CI4 file upload
     * Supports both local server and AWS S3 storage based on disk configuration
     * 
     * @return array - Upload result with success/error status
     */
    private function handleImageUpload()
    {
        $image = $this->request->getFile('image');

        // Check if image is provided and valid
        if (!$image || !$image->isValid() || $image->hasMoved()) {
            return [
                'error' => true,
                'message' => labels(INVALID_IMAGE_FILE, 'Invalid image file')
            ];
        }

        // Validate image file type using CI4 validation
        $validationRules = [
            'image' => [
                'rules' => 'uploaded[image]|is_image[image]|mime_in[image,image/jpg,image/jpeg,image/png]|max_size[image,5120]',
                'errors' => [
                    'uploaded' => labels(PLEASE_UPLOAD_AN_IMAGE_FILE, 'Please upload an image file'),
                    'is_image' => labels(FILE_MUST_BE_A_VALID_IMAGE, 'File must be a valid image'),
                    'mime_in' => labels(ONLY_JPEG_JPG_AND_PNG_FILES_ARE_ALLOWED, 'Only JPEG, JPG, and PNG files are allowed'),
                    'max_size' => labels(IMAGE_FILE_SIZE_MUST_NOT_EXCEED_5MB, 'Image file size must not exceed 5MB')
                ]
            ]
        ];

        if (!$this->validate($validationRules)) {
            $errors = $this->validator->getErrors();
            return [
                'error' => true,
                'message' => reset($errors) // Get first error message
            ];
        }

        // Use the project's upload helper function which handles both local and AWS S3
        $uploadResult = upload_file(
            $image,
            'public/uploads/seo_settings/general_seo_settings',
            labels(FAILED_TO_UPLOAD_IMAGE, 'Failed to upload image'),
            'seo_settings'
        );

        // Check if upload was successful
        if ($uploadResult['error']) {
            return [
                'error' => true,
                'message' => $uploadResult['message'] ?? labels(FAILED_TO_UPLOAD_IMAGE, 'Failed to upload image')
            ];
        }

        // Handle file path based on disk type
        $fileName = $uploadResult['file_name'];
        $disk = $uploadResult['disk'];

        return [
            'error' => false,
            'file_name' => $fileName,
            'disk' => $disk,
            'message' => labels(FILE_UPLOAD_SUCCESSFUL, 'File uploaded successfully')
        ];
    }

    /**
     * Clean up image file when record is deleted
     * Supports both local server and AWS S3 storage based on disk configuration
     * 
     * @param string $imageName - Name of the image file to delete
     * @return void
     */
    private function cleanupImageFile($imageName)
    {
        if (empty($imageName)) {
            return;
        }

        try {
            // Get current disk type from settings
            $disk = fetch_current_file_manager();

            // For local server, extract just the filename from the full path
            // For AWS S3, use the image name as is
            if ($disk === 'local_server') {
                // If the imageName contains the full path, extract just the filename
                if (strpos($imageName, 'public/uploads/seo_settings/general_seo_settings/') === 0) {
                    $fileName = basename($imageName);
                } else {
                    $fileName = $imageName;
                }
            } else {
                // For AWS S3, use the image name as stored
                $fileName = $imageName;
            }

            // Use the project's delete helper function which handles both local and AWS S3
            $result = delete_file_based_on_server('seo_settings/general_seo_settings', $fileName, $disk);

            if (!$result['error']) {
                log_message('info', 'SEO image file deleted successfully: ' . $fileName);
            } else {
                log_message('error', 'Failed to delete SEO image file: ' . $result['message']);
            }
        } catch (\Exception $e) {
            log_message('error', 'Failed to cleanup SEO image file: ' . $e->getMessage());
        }
    }

    /**
     * Generate operation buttons (Edit/Delete) for each SEO setting record
     * Using CSS classes for Bootstrap Table events instead of onclick
     * 
     * @param int $id - Record ID
     * @return string - HTML for operation buttons
     */
    private function generateOperationButtons($id, $permissions)
    {
        $operations = '<div class="dropdown">
        <a class="" href="#" role="button" id="dropdownMenuLink_' . $id . '" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
            <button class="btn btn-secondary btn-sm px-3">
                <i class="fas fa-ellipsis-v"></i>
            </button>
        </a>
        <div class="dropdown-menu" aria-labelledby="dropdownMenuLink_' . $id . '">';

        // Edit button
        if (!empty($permissions['update']) && $permissions['update'] == 1) {
            $operations .= '<a class="dropdown-item edit_seo_setting" data-id="' . $id . '" title="Edit">
            <i class="fa fa-pen text-primary mr-1" aria-hidden="true"></i>' . labels('edit', 'Edit') . '</a>';
        }

        // Delete button
        if (!empty($permissions['delete']) && $permissions['delete'] == 1) {
            $operations .= '<a class="dropdown-item delete_seo_setting" data-id="' . $id . '">
            <i class="fa fa-trash text-danger mr-1" aria-hidden="true"></i>' . labels('delete', 'Delete') . '</a>';
        }

        $operations .= '</div></div>';

        return $operations;
    }
}
