<?php

namespace App\Controllers\admin;

use App\Models\Service_model;
use App\Models\Language_model;
use App\Models\TranslatedServiceDetails_model;
use Config\ApiResponseAndNotificationStrings;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Models\Seo_model;
use App\Services\ServicesService;
use Exception;

class Services extends Admin
{
    public $validation, $db, $ionAuth, $creator_id, $service, $seoModel;
    protected $superadmin;
    protected ApiResponseAndNotificationStrings $trans;
    protected ServicesService $serviceService;

    public function __construct()
    {
        parent::__construct();
        $this->service = new Service_model();
        $this->seoModel = new Seo_model();
        $this->validation = \Config\Services::validation();
        $this->db = \Config\Database::connect();
        $this->ionAuth = new \IonAuth\Libraries\IonAuth();
        $this->creator_id = $this->userId;
        $this->superadmin = $this->session->get('email');
        $this->trans = new ApiResponseAndNotificationStrings();
        $this->serviceService = new ServicesService();
        helper('ResponceServices');
    }

    public function index()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            setPageInfo($this->data, labels('services', 'Services') . ' | ' . labels('admin_panel', 'Admin Panel'), 'services');
            // $this->data['categories_name'] = fetch_details('categories', [], ['id', 'name', 'parent_id']);
            $this->data['categories_name'] = get_categories_with_translated_names();
            $this->data['categories_tree'] = $this->getCategoriesTree();

            // Get current language for translation
            $currentLanguage = get_current_language();

            // Fetch partner data with translated company names based on current language
            $partner_data = $this->db->table('users u')
                ->select('u.id,u.username,pd.company_name,pd.number_of_members,tpd.company_name as translated_company_name')
                ->join('partner_details pd', 'pd.partner_id = u.id')
                ->join('translated_partner_details tpd', 'tpd.partner_id = pd.partner_id AND tpd.language_code = "' . $currentLanguage . '"', 'left')
                ->where('pd.is_approved', '1')
                ->get()->getResultArray();

            // Process partner data to use translated company names when available
            foreach ($partner_data as &$partner) {
                // Use translated company name if available and not empty, otherwise use original
                if (!empty($partner['translated_company_name'])) {
                    $partner['display_company_name'] = $partner['translated_company_name'];
                } else {
                    $partner['display_company_name'] = $partner['company_name'];
                }
            }

            $this->data['partner_name'] = $partner_data;
            // Fetch taxes with translated names based on current language
            $tax_data = get_taxes_with_translated_names(['status' => 1], ['id', 'title', 'percentage']);
            $this->data['tax_data'] = $tax_data;
            return view('backend/admin/template', $this->data);
        } catch (\Throwable $th) {
            throw $th;
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - index()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    function getCategoriesTree()
    {
        try {
            $categories = $this->db->table('categories')->get()->getResultArray();
            $tree = [];
            foreach ($categories as $category) {
                if (!$category['parent_id']) {
                    $tree[] = $this->buildTree($categories, $category);
                }
            }
            return $tree;
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - getCategoriesTree()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    function buildTree(&$categories, $currentCategory)
    {
        try {
            $tree = [
                'id' => $currentCategory['id'],
                'text' => $currentCategory['name'],
            ];
            $children = [];
            foreach ($categories as $category) {
                if ($category['parent_id'] == $currentCategory['id']) {
                    $children[] = $this->buildTree($categories, $category);
                }
            }
            if (!empty($children)) {
                $tree['children'] = $children;
            }
            return $tree;
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - buildTree()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function list($from_app = false, $search = '', $limit = 10, $offset = 0, $sort = 'id', $order = 'ASC', $where = [], $additional_data = [], $column_name = '', $whereIn = [])
    {
        try {
            $Service_model = new Service_model();
            $limit = (isset($_GET['limit']) && !empty($_GET['limit'])) ? $_GET['limit'] : 10;
            $offset = (isset($_GET['offset']) && !empty($_GET['offset'])) ? $_GET['offset'] : 0;
            $sort = (isset($_GET['sort']) && !empty($_GET['sort'])) ? $_GET['sort'] : 'id';
            $order = (isset($_GET['order']) && !empty($_GET['order'])) ? $_GET['order'] : 'ASC';
            $search = (isset($_GET['search']) && !empty($_GET['search'])) ? $_GET['search'] : '';
            $data = $Service_model->list(false, $search, $limit, $offset, $sort, $order);
            return $data;
        } catch (\Throwable $th) {
            throw $th;
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - list()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function add_service()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }
            if (isset($_POST) && !empty($_POST)) {
                $price = $this->request->getPost('price');
                // Get the default language from database
                $defaultLanguage = 'en'; // fallback
                $languages = fetch_details('languages', [], ['id', 'language', 'code', 'is_default'], "", '0', 'id', 'ASC');
                foreach ($languages as $language) {
                    if ($language['is_default'] == 1) {
                        $defaultLanguage = $language['code'];
                        break;
                    }
                }

                // Validate default language fields
                $postData = $this->request->getPost();
                $defaultLanguageErrors = [];

                // Check if data is in new format (as objects)
                if (isset($postData['title']) && is_array($postData['title'])) {
                    // Check title
                    $titleValue = $postData['title'][$defaultLanguage] ?? null;
                    if (empty($titleValue)) {
                        $defaultLanguageErrors[] = labels("service_title_in_default_language_is_required", "Service title is required for default language");
                    }

                    // Check description
                    $descriptionValue = $postData['description'][$defaultLanguage] ?? null;
                    if (empty($descriptionValue)) {
                        $defaultLanguageErrors[] = labels("service_description_in_default_language_is_required", "Service description is required for default language");
                    }

                    // Check long description
                    $longDescriptionValue = $postData['long_description'][$defaultLanguage] ?? null;
                    if (empty($longDescriptionValue)) {
                        $defaultLanguageErrors[] = labels("service_long_description_in_default_language_is_required", "Service long description is required for default language");
                    }

                    // Check tags
                    $tagsValue = $this->processTagsValue($postData['tags'][$defaultLanguage] ?? null);
                    if (empty($tagsValue)) {
                        $defaultLanguageErrors[] = labels("service_tags_in_default_language_are_required", "Service tags are required for default language");
                    }
                } else {
                    // Fallback: Check old format (field[language])
                    $titleField = 'title[' . $defaultLanguage . ']';
                    $descriptionField = 'description[' . $defaultLanguage . ']';
                    $longDescriptionField = 'long_description[' . $defaultLanguage . ']';
                    $tagsField = 'tags[' . $defaultLanguage . ']';

                    // Check title
                    $titleValue = $postData[$titleField] ?? null;
                    if (empty($titleValue)) {
                        $defaultLanguageErrors[] = labels("service_title_in_default_language_is_required", "Service title is required for default language");
                    }

                    // Check description
                    $descriptionValue = $postData[$descriptionField] ?? null;
                    if (empty($descriptionValue)) {
                        $defaultLanguageErrors[] = labels("service_description_in_default_language_is_required", "Service description is required for default language");
                    }

                    // Check long description
                    $longDescriptionValue = $postData[$longDescriptionField] ?? null;
                    if (empty($longDescriptionValue)) {
                        $defaultLanguageErrors[] = labels("service_long_description_in_default_language_is_required", "Service long description is required for default language");
                    }

                    // Check tags
                    $tagsValue = $this->processTagsValue($postData[$tagsField] ?? null);
                    if (empty($tagsValue)) {
                        $defaultLanguageErrors[] = labels("service_tags_in_default_language_are_required", "Service tags are required for default language");
                    }
                }

                if (!empty($defaultLanguageErrors)) {
                    return ErrorResponse($defaultLanguageErrors, true, [], [], 200, csrf_token(), csrf_hash());
                }

                $this->validation->setRules(
                    [
                        'partner' => ["rules" => 'required', "errors" => ["required" => "Please select provider"]],
                        'categories' => ["rules" => 'required', "errors" => ["required" => "Please select category"]],
                        'price' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter price",     "numeric" => "Please enter numeric value for price"]],
                        'discounted_price' => ["rules" => 'required|numeric|less_than[' . $price . ']', "errors" => ["required" => "Please enter discounted price",    "numeric" => "Please enter numeric value for discounted price",    "less_than" => "Discounted price should be less than price"]],
                        'members' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter required member for service",    "numeric" => "Please enter numeric value for required member"]],
                        'duration' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter duration to perform task",    "numeric" => "Please enter numeric value for duration of task"]],
                        'max_qty' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter max quantity allowed for services",     "numeric" => "Please enter numeric value for max quantity allowed for services"]],
                        'meta_title' => ["rules" => 'permit_empty', "errors" => ["permit_empty" => "Meta title is optional"]],
                        'meta_description' => ["rules" => 'permit_empty', "errors" => ["permit_empty" => "Meta description is optional"]],
                        'meta_keywords' => ["rules" => 'permit_empty', "errors" => ["permit_empty" => "Meta keywords are optional"]],
                        'meta_image' => ["rules" => 'permit_empty|uploaded[meta_image]|is_image[meta_image]', "errors" => ["permit_empty" => "Meta image is optional", "uploaded" => "Invalid meta image", "is_image" => "Meta image must be a valid image"]],
                        'schema_markup' => ["rules" => 'permit_empty', "errors" => ["permit_empty" => "Schema markup is optional"]],
                        'service_slug' => ["rules" => 'required', "errors" => ["required" => "Please enter service slug",]],
                    ],
                );
                if (!$this->validation->withRequest($this->request)->run()) {
                    $errors  = $this->validation->getErrors();
                    return ErrorResponse($errors, true, [], [], 200, csrf_token(), csrf_hash());
                }

                // Get default language fields for main table (for slug generation and fallback)
                $defaultTitle = '';
                if (isset($postData['title']) && is_array($postData['title'])) {
                    $defaultTitle = $postData['title'][$defaultLanguage] ?? '';
                } else {
                    $defaultTitle = $postData['title[' . $defaultLanguage . ']'] ?? '';
                }

                $path = "public/uploads/services/";

                // Check if this is a cloning operation (service_id exists)
                $cloning = false;
                $source_service_id = $this->request->getVar('service_id');
                $original_service = null;

                if ($source_service_id) {
                    $original_service = fetch_details('services', ['id' => $source_service_id], ['image', 'other_images', 'files']);
                    if (!empty($original_service)) {
                        $cloning = true;
                    }
                }

                if (!is_dir(FCPATH . 'public/uploads/services/')) {
                    if (!mkdir(FCPATH . 'public/uploads/services/', 0775, true)) {
                        return ErrorResponse("Failed to create folders", true, [], [], 200, csrf_token(), csrf_hash());
                    }
                }
                $paths = [
                    'image' => [
                        'file' => $this->request->getFile('service_image_selector'),
                        'path' => 'public/uploads/services/',
                        'error' => labels('Failed to create services folders', 'Failed to create services folders'),
                        'folder' => 'services'
                    ],
                ];
                $uploadedFiles = [];


                foreach ($paths as $key => $upload) {
                    if ($upload['file'] && $upload['file']->isValid()) {
                        $result = upload_file($upload['file'], $upload['path'], $upload['error'], $upload['folder']);
                        if ($result['error'] == false) {
                            $uploadedFiles[$key] = [
                                'url' => $result['file_name'],
                                'disk' => $result['disk']
                            ];
                        } else {
                            return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
                        }
                    }
                }
                $multipleFiles = $this->request->getFiles('filepond');
                $otherImagesConfig = [
                    'path' => 'public/uploads/services/',
                    'error' => labels('Failed to upload other images', 'Failed to upload other images'),
                    'folder' => 'services'
                ];
                $uploadedOtherImages = [];
                if (isset($multipleFiles['other_service_image_selector'])) {
                    $files = $multipleFiles['other_service_image_selector'];
                    foreach ($files as $file) {
                        if (!empty($files[0]) && $files[0]->getSize() > 0) {

                            if ($file->isValid()) {
                                $result = upload_file($file, $otherImagesConfig['path'], $otherImagesConfig['error'], $otherImagesConfig['folder']);
                                if ($result['error'] == false) {
                                    if ($result['disk'] == "local_server") {
                                        $uploadedOtherImages[] = 'public/uploads/services/' . $result['file_name'];
                                    } elseif ($result['disk'] == "aws_s3") {
                                        $uploadedOtherImages[] = $result['file_name'];
                                    } else {
                                        $uploadedOtherImages[] = $result['file_name'];
                                    }
                                } else {
                                    return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
                                }
                            }
                        } else if ($cloning && empty($uploadedOtherImages) && !empty($original_service[0]['other_images'])) {
                            // If cloning and no new images uploaded, use original images
                            $other_images_data = json_decode($original_service[0]['other_images'], true);
                            if (is_array($other_images_data)) {
                                $uploadedOtherImages = $other_images_data;
                            }
                            break;
                        }
                    }
                } else if ($cloning && !empty($original_service[0]['other_images'])) {
                    // If cloning and no file input, use original images
                    $other_images_data = json_decode($original_service[0]['other_images'], true);
                    if (is_array($other_images_data)) {
                        $uploadedOtherImages = $other_images_data;
                    }
                }

                // Process existing other images that the user wants to keep or remove
                $existing_other_images = $this->request->getPost('existing_other_images');
                $remove_other_images_flags = $this->request->getPost('remove_other_images');

                // If the user has marked images for removal
                if ($cloning && !empty($existing_other_images) && !empty($remove_other_images_flags)) {
                    // Create a clean array for images to keep
                    $filtered_other_images = [];

                    foreach ($existing_other_images as $index => $image_path) {
                        // Skip images marked for removal
                        if (isset($remove_other_images_flags[$index]) && $remove_other_images_flags[$index] === "1") {
                            continue;
                        }

                        // Keep only the path part without base_url
                        $base_url = base_url();
                        if (strpos($image_path, $base_url) === 0) {
                            $image_path = substr($image_path, strlen($base_url));
                        }

                        $filtered_other_images[] = $image_path;
                    }

                    // Update the uploaded other images with only the ones to keep
                    $uploadedOtherImages = $filtered_other_images;
                }

                $other_images = [
                    'other_images' => !empty($uploadedOtherImages) ? json_encode($uploadedOtherImages) : "[]",
                ];
                $uploadedFilesDocuments = [];
                $FilesDocumentsConfig = [
                    'path' => 'public/uploads/services/',
                    'error' => labels('Failed to upload files', 'Failed to upload files'),
                    'folder' => 'services'
                ];
                if (isset($multipleFiles['files'])) {
                    $files = $multipleFiles['files'];

                    if (!empty($files[0]) && $files[0]->getSize() > 0) {
                        foreach ($files as $file) {
                            if ($file->isValid()) {
                                $result = upload_file($file, $FilesDocumentsConfig['path'], $FilesDocumentsConfig['error'], $FilesDocumentsConfig['folder']);
                                if ($result['error'] == false) {
                                    if ($result['disk'] == "local_server") {
                                        $uploadedFilesDocuments[] = 'public/uploads/services/' . $result['file_name'];
                                    } elseif ($result['disk'] == "aws_s3") {
                                        $uploadedFilesDocuments[] = $result['file_name'];
                                    } else {
                                        $uploadedFilesDocuments[] = $result['file_name'];
                                    }
                                } else {
                                    return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
                                }
                            }
                        }
                    } else if ($cloning && empty($uploadedFilesDocuments) && !empty($original_service[0]['files'])) {
                        // If cloning and no new files uploaded, use original files
                        $files_data = json_decode($original_service[0]['files'], true);
                        if (is_array($files_data)) {
                            $uploadedFilesDocuments = $files_data;
                        }
                    }
                } else if ($cloning && !empty($original_service[0]['files'])) {
                    // If cloning and no file input, use original files
                    $files_data = json_decode($original_service[0]['files'], true);
                    if (is_array($files_data)) {
                        $uploadedFilesDocuments = $files_data;
                    }
                }

                // Process existing files that the user wants to keep or remove
                $existing_files = $this->request->getPost('existing_files');
                $remove_files_flags = $this->request->getPost('remove_files');

                // If the user has marked files for removal
                if ($cloning && !empty($existing_files) && !empty($remove_files_flags)) {
                    // Create a clean array for files to keep
                    $filtered_files = [];

                    foreach ($existing_files as $index => $file_path) {
                        // Skip files marked for removal
                        if (isset($remove_files_flags[$index]) && $remove_files_flags[$index] === "1") {
                            continue;
                        }

                        // Keep only the path part without base_url
                        $base_url = base_url();
                        if (strpos($file_path, $base_url) === 0) {
                            $file_path = substr($file_path, strlen($base_url));
                        }

                        $filtered_files[] = $file_path;
                    }

                    // Update the uploaded files with only the ones to keep
                    $uploadedFilesDocuments = $filtered_files;
                }

                $files = !empty($uploadedFilesDocuments) ? json_encode($uploadedFilesDocuments) : "[]";

                $category_id = $this->request->getPost('categories');
                $discounted_price = $this->request->getPost('discounted_price');
                if ($discounted_price >= $price && $discounted_price == $price) {
                    return ErrorResponse("discounted price can not be higher than or equal to the price!", true, [], [], 200, csrf_token(), csrf_hash());
                }
                $user_id = $this->request->getPost('partner');
                $partner_data = fetch_details('partner_details', ['partner_id' => $this->request->getPost('partner')]);
                if ($this->request->getVar('members') > $partner_data[0]['number_of_members']) {
                    return ErrorResponse("Number Of member could not greater than " . $partner_data[0]['number_of_members'], true, [], [], 200, csrf_token(), csrf_hash());
                }
                // FAQs are now handled as translatable fields, so we don't need to process them here
                // The ServiceService will handle FAQ storage in the translated_service_details table
                $check_payment_gateway = get_settings('payment_gateways_settings', true);
                $cod_setting =  $check_payment_gateway['cod_setting'];
                if ($cod_setting == 1) {
                    $is_pay_later_allowed = ($this->request->getPost('pay_later') == "on") ? 1 : 0;
                } else {
                    $is_pay_later_allowed = 0;
                }
                $is_cancelable = (isset($_POST['is_cancelable'])) ? 1 : 0;
                if (isset($uploadedFiles['image']['disk']) && $uploadedFiles['image']['disk'] == 'local_server') {
                    $image = 'public/uploads/services/' . $uploadedFiles['image']['url'];
                } else {
                    if ($cloning && !empty($original_service[0]['image'])) {
                        $image = $original_service[0]['image'];
                    } else {
                        $image = $uploadedFiles['image']['url'] ?? "";
                    }
                }

                // Get default language field values for main table storage
                $defaultTitle = $titleValue ?? '';
                $defaultDescription = $descriptionValue ?? '';
                $defaultLongDescription = $longDescriptionValue ?? '';
                $defaultTags = $tagsValue ?? '';

                // Process default language FAQs for main table
                $defaultFaqs = '';
                if (isset($postData['faqs'])) {
                    // Check if FAQs are in JSON string format
                    if (is_string($postData['faqs'])) {
                        $faqsData = json_decode($postData['faqs'], true);
                        if (json_last_error() === JSON_ERROR_NONE && is_array($faqsData)) {
                            $defaultFaqsData = $faqsData[$defaultLanguage] ?? [];
                            if (!empty($defaultFaqsData)) {
                                $defaultFaqs = json_encode($defaultFaqsData, JSON_UNESCAPED_UNICODE);
                            }
                        }
                    } elseif (is_array($postData['faqs'])) {
                        $defaultFaqsData = $postData['faqs'][$defaultLanguage] ?? [];
                        if (!empty($defaultFaqsData)) {
                            $defaultFaqs = json_encode($defaultFaqsData, JSON_UNESCAPED_UNICODE);
                        }
                    }
                }

                // Prepare service data WITH default language translatable fields in main table
                $service = [
                    'user_id' => $user_id,
                    'category_id' => $category_id,
                    'tax_type' => $this->request->getVar('tax_type'),
                    'tax_id' => $this->request->getVar('tax_id'),
                    'price' => $price,
                    'discounted_price' => $discounted_price,
                    'image' => $image,
                    'other_images' => $other_images['other_images'] ?? '',
                    'number_of_members_required' => $this->request->getVar('members'),
                    'duration' => $this->request->getVar('duration'),
                    'rating' => 0,
                    'number_of_ratings' => 0,
                    'on_site_allowed' => ($this->request->getPost('on_site') == "on") ? 1 : 0,
                    'is_pay_later_allowed' => $is_pay_later_allowed,
                    'is_cancelable' => $is_cancelable,
                    'cancelable_till' => $this->request->getVar('cancelable_till'),
                    'max_quantity_allowed' => $this->request->getPost('max_qty'),
                    'status' => (isset($_POST['status'])) ? 1 : 0,
                    'files' => $files,
                    'at_store' => (isset($_POST['at_store'])) ? 1 : 0,
                    'at_doorstep' => (isset($_POST['at_doorstep'])) ? 1 : 0,
                    'approved_by_admin' => isset($_POST['approve_service_value']) ? $_POST['approve_service_value'] : "1",
                    // Use default language title for slug generation
                    'slug' => generate_unique_slug($this->request->getPost('service_slug'), 'services'),
                    // Store default language translations in main table as well
                    'title' => $defaultTitle,
                    'description' => $defaultDescription,
                    'long_description' => $defaultLongDescription,
                    'tags' => $defaultTags,
                    'faqs' => $defaultFaqs,
                ];

                if ($this->service->save($service)) {
                    $serviceId = $this->service->insertID(); // Get the inserted service ID

                    // Handle translated fields using ServiceService
                    $postData = $this->request->getPost();

                    // Transform form data to translated_fields structure
                    $translatedFields = $this->transformFormDataToTranslatedFields($postData, $defaultLanguage, null);

                    // Add translated_fields to postData for ServiceService
                    $postData['translated_fields'] = $translatedFields;

                    $translationResult = $this->serviceService->handleServiceCreationWithTranslations($postData, $service, $serviceId, $defaultLanguage);

                    if (!$translationResult['success']) {
                        // If translation saving fails, we should handle this appropriately
                        // For now, we'll log the error but continue with the process
                        log_message('error', 'Failed to save service translations: ' . implode(', ', $translationResult['errors']));
                    }

                    try {
                        $this->saveServiceSeoSettings($serviceId); // Save SEO settings
                    } catch (\Throwable $th) {
                        log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - add_service() - SEO settings');
                        return ErrorResponse(labels(ERROR_OCCURED, "Failed to save SEO settings: " . $th->getMessage()), true, [], [], 200, csrf_token(), csrf_hash());
                    }
                    $message = $cloning ? labels("service_cloned_successfully", "Service cloned successfully") : labels(DATA_SAVED_SUCCESSFULLY, "Service saved successfully");
                    return successResponse($message, false, [], [], 200, csrf_token(), csrf_hash());
                } else {
                    return ErrorResponse(labels(ERROR_OCCURED, "Service can not be saved!"), true, [], [], 200, csrf_token(), csrf_hash());
                }
            } else {
                return redirect()->to('partner/services');
            }
        } catch (\Throwable $th) {
            throw $th;
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - add_service()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function delete_service()
    {
        try {
            $disk = fetch_current_file_manager();

            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }
            $id = $this->request->getPost('id');
            $old_data = fetch_details('services', ['id' => $id]);
            if ($old_data[0]['image'] != NULL &&  !empty($old_data[0]['image'])) {
                delete_file_based_on_server('services', $old_data[0]['image'], $disk);
            }
            if ($old_data[0]['other_images'] != NULL &&  !empty($old_data[0]['other_images'])) {
                $other_images = json_decode($old_data[0]['other_images'], true);
                foreach ($other_images as $oi) {
                    delete_file_based_on_server('services', $oi, $disk);
                }
            }
            if ($old_data[0]['files'] != NULL &&  !empty($old_data[0]['files'])) {
                $files = json_decode($old_data[0]['files'], true);
                foreach ($files as $oi) {
                    delete_file_based_on_server('services', $oi, $disk);
                }
            }

            // Clean up SEO settings and images before deleting service
            $this->seoModel->cleanupSeoData($id, 'services');

            // Clean up service translations before deleting service
            $this->serviceService->deleteServiceTranslations($id);

            $builder = $this->db->table('services')->delete(['id' => $id]);
            $builder2 = $this->db->table('cart')->delete(['service_id' => $id]);
            $builder3 = $this->db->table('services_ratings')->delete(['service_id' => $id]);
            if ($builder) {
                return successResponse("success in deleting the service", false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return ErrorResponse("Unsuccessful in deleting services", true, [], [], 200, csrf_token(), csrf_hash());
            }
        } catch (\Throwable $th) {

            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - delete_service()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function update_service()
    {
        try {

            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }

            if ($_POST && !empty($_POST)) {
                $disk = fetch_current_file_manager();

                $price = $this->request->getPost('price');

                // Get the default language from database
                $defaultLanguage = 'en'; // fallback
                $languages = fetch_details('languages', [], ['id', 'language', 'code', 'is_default'], "", '0', 'id', 'ASC');
                foreach ($languages as $language) {
                    if ($language['is_default'] == 1) {
                        $defaultLanguage = $language['code'];
                        break;
                    }
                }

                // Validate default language fields
                $postData = $this->request->getPost();
                $defaultLanguageErrors = [];

                // Check if data is in new format (as objects)
                if (isset($postData['title']) && is_array($postData['title'])) {
                    // Check title
                    $titleValue = $postData['title'][$defaultLanguage] ?? null;
                    if (empty($titleValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_TITLE_IN_DEFAULT_LANGUAGE_IS_REQUIRED, "Service title in default language is required!");
                    }

                    // Check description
                    $descriptionValue = $postData['description'][$defaultLanguage] ?? null;
                    if (empty($descriptionValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_DESCRIPTION_IN_DEFAULT_LANGUAGE_IS_REQUIRED, "Service description in default language is required!");
                    }

                    // Check long description
                    $longDescriptionValue = $postData['long_description'][$defaultLanguage] ?? null;
                    if (empty($longDescriptionValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_LONG_DESCRIPTION_IN_DEFAULT_LANGUAGE_IS_REQUIRED, "Service long description in default language is required!");
                    }

                    // Check tags
                    $tagsValue = $postData['tags'][$defaultLanguage] ?? null;
                    if (empty($tagsValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_TAGS_IN_DEFAULT_LANGUAGE_ARE_REQUIRED, "Service tags in default language are required!");
                    }
                } else {
                    // Fallback: Check old format (field[language])
                    $titleField = 'title[' . $defaultLanguage . ']';
                    $descriptionField = 'description[' . $defaultLanguage . ']';
                    $longDescriptionField = 'long_description[' . $defaultLanguage . ']';
                    $tagsField = 'tags[' . $defaultLanguage . ']';

                    // Check title
                    $titleValue = $postData[$titleField] ?? null;
                    if (empty($titleValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_TITLE_IN_DEFAULT_LANGUAGE_IS_REQUIRED, "Service title in default language is required!");
                    }

                    // Check description
                    $descriptionValue = $postData[$descriptionField] ?? null;
                    if (empty($descriptionValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_DESCRIPTION_IN_DEFAULT_LANGUAGE_IS_REQUIRED, "Service description in default language is required!");
                    }

                    // Check long description
                    $longDescriptionValue = $postData[$longDescriptionField] ?? null;
                    if (empty($longDescriptionValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_LONG_DESCRIPTION_IN_DEFAULT_LANGUAGE_IS_REQUIRED, "Service long description in default language is required!");
                    }

                    // Check tags
                    $tagsValue = $postData[$tagsField] ?? null;
                    if (empty($tagsValue)) {
                        $defaultLanguageErrors[] = labels(SERVICE_TAGS_IN_DEFAULT_LANGUAGE_ARE_REQUIRED, "Service tags in default language are required!");
                    }
                }

                if (!empty($defaultLanguageErrors)) {
                    return ErrorResponse($defaultLanguageErrors, true, [], [], 200, csrf_token(), csrf_hash());
                }

                $rules = [
                    'partner' => ["rules" => 'required', "errors" => ["required" => "Please select provider"]],
                    'categories' => ["rules" => 'required', "errors" => ["required" => "Please select category"]],
                    'price' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter price",     "numeric" => "Please enter numeric value for price"]],
                    'discounted_price' => ["rules" => 'required|numeric|less_than[' . $price . ']', "errors" => ["required" => "Please enter discounted price",    "numeric" => "Please enter numeric value for discounted price",    "less_than" => "Discounted price should be less than price"]],
                    'members' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter required member for service",    "numeric" => "Please enter numeric value for required member"]],
                    'duration' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter duration to perform task",    "numeric" => "Please enter numeric value for duration of task"]],
                    'max_qty' => ["rules" => 'required|numeric', "errors" => ["required" => "Please enter max quantity allowed for services",     "numeric" => "Please enter numeric value for max quantity allowed for services"]],

                ];
                if (isset($_FILES['service_image_selector']) && $_FILES['service_image_selector']['size'] > 0) {
                    $rules['service_image_selector'] = [
                        "rules" => 'uploaded[service_image_selector]|ext_in[service_image_selector,png,jpg,gif,jpeg,webp]|max_size[service_image_selector,8496]|is_image[service_image_selector]'
                    ];
                }
                $this->validation->setRules($rules);
                if (!$this->validation->withRequest($this->request)->run()) {
                    $errors = $this->validation->getErrors();
                    return ErrorResponse($errors, true, [], [], 200, csrf_token(), csrf_hash());
                }
                $Service_id = $this->request->getPost('service_id');
                $old_images_and_documents = fetch_details('services', ['id' => $Service_id]);

                // Process default language tags
                $defaultTags = $this->request->getPost('tags[' . $defaultLanguage . ']') ?: $this->request->getPost('tags');
                if (is_array($defaultTags)) {
                    $tags = [];
                    foreach ($defaultTags as $tag) {
                        if (is_string($tag)) {
                            $tags[] = trim($tag);
                        } elseif (is_array($tag) && isset($tag['value'])) {
                            $tags[] = trim($tag['value']);
                        }
                    }
                } else {
                    // Handle string format tags
                    $tags = explode(',', $defaultTags);
                    $tags = array_map('trim', $tags);
                    $tags = array_filter($tags);
                }

                if (empty($tags)) {
                    return ErrorResponse("Service tags in default language are required!", true, [], [], 200, csrf_token(), csrf_hash());
                }
                // FAQs are now handled as translatable fields, so we don't need to process them here
                // The ServiceService will handle FAQ storage in the translated_service_details table
                $paths = [
                    'image' => [
                        'file' => $this->request->getFile('service_image_selector_edit'),
                        'path' => 'public/uploads/services/',
                        'error' => labels('Failed to create services folders', 'Failed to create services folders'),
                        'folder' => 'services',
                        'old_file' => $old_images_and_documents[0]['image'],
                        'disk' => $disk,
                    ],
                ];
                $uploadedFiles = [];
                foreach ($paths as $key => $upload) {
                    if ($upload['file'] && $upload['file']->isValid()) {
                        if (!empty($upload['old_file'])) {
                            delete_file_based_on_server($upload['folder'], $upload['old_file'], $upload['disk']);
                        }
                        $result = upload_file($upload['file'], $upload['path'], $upload['error'], $upload['folder']);
                        if ($result['error'] === false) {
                            $uploadedFiles[$key] = [
                                'url' => $result['file_name'],
                                'disk' => $result['disk']
                            ];
                        } else {
                            return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
                        }
                    }
                }

                if (isset($uploadedFiles['image']['disk']) && $uploadedFiles['image']['disk'] == 'local_server') {
                    $image_name = isset($uploadedFiles['image']['url']) ? ('public/uploads/services/' . $uploadedFiles['image']['url']) : $old_images_and_documents[0]['image'];
                } else {
                    $image_name = isset($uploadedFiles['image']['url']) ? $uploadedFiles['image']['url'] : $old_images_and_documents[0]['image'];
                }

                // Initialize arrays for images
                $updated_images = [];
                $uploadedOtherImages = [];
                $uploadedFilesDocuments = [];

                // Process existing images and check for removals
                $existing_images = $this->request->getPost('existing_other_images');
                $remove_flags = $this->request->getPost('remove_other_images');

                // Strip base URL from existing image paths to get correct relative paths
                if (!empty($existing_images)) {
                    foreach ($existing_images as $key => $image_path) {
                        // Remove base URL if present
                        $base_url = base_url();
                        if (strpos($image_path, $base_url) === 0) {
                            $existing_images[$key] = substr($image_path, strlen($base_url));
                        }
                    }
                }

                // First handle existing images (if any) and process removals
                if (!empty($existing_images)) {
                    foreach ($existing_images as $index => $image) {
                        // Check if this image is marked for removal
                        if (isset($remove_flags[$index]) && $remove_flags[$index] === "1") {
                            // Delete the image marked for removal
                            delete_file_based_on_server('services', $image, $disk);
                        } else {
                            // Keep images not marked for removal
                            $updated_images[] = $image;
                        }
                    }
                } else if (!empty($old_images_and_documents[0]['other_images']) && !$this->request->getPost('remove_other_images')) {
                    // If no form interaction but old images exist in DB
                    $old_other_images = json_decode($old_images_and_documents[0]['other_images'], true);
                    if (!empty($old_other_images) && is_array($old_other_images)) {
                        $updated_images = $old_other_images;
                    }
                }

                // Now handle new image uploads and add them to the updated_images array
                $multipleFiles = $this->request->getFiles('filepond');
                if (isset($multipleFiles['other_service_image_selector_edit'])) {
                    foreach ($multipleFiles['other_service_image_selector_edit'] as $file) {
                        if ($file->isValid()) {
                            // Upload new image
                            $result = upload_file($file, 'public/uploads/services/', labels('Failed to upload other images', 'Failed to upload other images'), 'services');
                            if ($result['error'] == false) {
                                $new_image = $result['disk'] === "local_server"
                                    ? 'public/uploads/services/' . $result['file_name']
                                    : $result['file_name'];

                                // Add new image to our list
                                $updated_images[] = $new_image;
                            } else {
                                return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
                            }
                        }
                    }
                }

                // Set the other_images field with all preserved existing images and new uploads
                $other_images = !empty($updated_images) ? json_encode($updated_images) : "[]";

                // Process existing files and check for removals
                $updated_files = [];
                $existing_files = $this->request->getPost('existing_files');
                $remove_files_flags = $this->request->getPost('remove_files');

                // Strip base URL from existing file paths to get correct relative paths
                if (!empty($existing_files)) {
                    foreach ($existing_files as $key => $file_path) {
                        // Remove base URL if present
                        $base_url = base_url();
                        if (strpos($file_path, $base_url) === 0) {
                            $existing_files[$key] = substr($file_path, strlen($base_url));
                        }
                    }
                }

                // First handle existing files (if any) and process removals
                if (!empty($existing_files)) {
                    foreach ($existing_files as $index => $file) {
                        // Check if this file is marked for removal
                        if (isset($remove_files_flags[$index]) && $remove_files_flags[$index] === "1") {
                            // Delete the file marked for removal
                            delete_file_based_on_server('services', $file, $disk);
                        } else {
                            // Keep files not marked for removal
                            $updated_files[] = $file;
                        }
                    }
                } else if (!empty($old_images_and_documents[0]['files']) && !$this->request->getPost('remove_files')) {
                    // If no form interaction but old files exist in DB
                    $old_files = json_decode($old_images_and_documents[0]['files'], true);
                    if (!empty($old_files) && is_array($old_files)) {
                        $updated_files = $old_files;
                    }
                }

                // Handle Files Upload
                if (isset($multipleFiles['files_edit'])) {
                    foreach ($multipleFiles['files_edit'] as $file) {
                        if ($file->isValid()) {
                            // Upload new file
                            $result = upload_file($file, 'public/uploads/services/', labels('Failed to upload files', 'Failed to upload files'), 'services');
                            if ($result['error'] == false) {
                                $new_file = $result['disk'] === "local_server"
                                    ? 'public/uploads/services/' . $result['file_name']
                                    : $result['file_name'];

                                // Add new file to our list
                                $updated_files[] = $new_file;
                            } else {
                                return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
                            }
                        }
                    }
                }

                // Set the files field with all preserved existing files and new uploads
                $files = !empty($updated_files) ? json_encode($updated_files) : "[]";

                $discounted_price = $this->request->getPost('discounted_price');
                if ($discounted_price >= $price) {
                    return ErrorResponse("Discounted price cannot be higher than or equal to the price", true, [], [], 200, csrf_token(), csrf_hash());
                }
                if (isset($_POST['is_cancelable']) && $_POST['is_cancelable'] == 'on') {
                    $is_cancelable = "1";
                } else {
                    $is_cancelable = "0";
                }
                if ($is_cancelable == "1" && $this->request->getVar('cancelable_till') == "") {
                    return ErrorResponse("Please Add Minutes", true, [], [], 200, csrf_token(), csrf_hash());
                }


                $check_payment_gateway = get_settings('payment_gateways_settings', true);
                $cod_setting =  $check_payment_gateway['cod_setting'];
                if ($cod_setting == 1) {
                    $is_pay_later_allowed = ($this->request->getPost('pay_later') == "on") ? 1 : 0;
                } else {
                    $is_pay_later_allowed = 0;
                }

                // Store default language data in main table
                $defaultTitle = $titleValue ?? '';
                $defaultDescription = $descriptionValue ?? '';
                $defaultLongDescription = $longDescriptionValue ?? '';
                $defaultTags = $this->processTagsValue($tagsValue) ?? '';

                // Process default language FAQs for main table
                $defaultFaqs = '';
                if (isset($postData['faqs'])) {
                    // Check if FAQs are in JSON string format
                    if (is_string($postData['faqs'])) {
                        $faqsData = json_decode($postData['faqs'], true);
                        if (json_last_error() === JSON_ERROR_NONE && is_array($faqsData)) {
                            $defaultFaqsData = $faqsData[$defaultLanguage] ?? [];
                            if (!empty($defaultFaqsData)) {
                                $defaultFaqs = json_encode($defaultFaqsData, JSON_UNESCAPED_UNICODE);
                            }
                        }
                    } elseif (is_array($postData['faqs'])) {
                        $defaultFaqsData = $postData['faqs'][$defaultLanguage] ?? [];
                        if (!empty($defaultFaqsData)) {
                            $defaultFaqs = json_encode($defaultFaqsData, JSON_UNESCAPED_UNICODE);
                        }
                    }
                }

                $service = [
                    'user_id' => $this->request->getPost('partner'),
                    'category_id' => $_POST['categories'],
                    'tax_type' => $this->request->getPost('tax_type'),
                    'tax_id' => $this->request->getPost('tax_id'),
                    'tax' => $this->request->getPost('tax'),
                    'price' => $price,
                    'discounted_price' => $discounted_price,
                    'image' => $image_name,
                    'number_of_members_required' => $this->request->getPost('members'),
                    'duration' => $this->request->getPost('duration'),
                    'max_quantity_allowed' => $this->request->getPost('max_qty'),
                    'status' => ($this->request->getPost('status') === "on") ? 1 : 0,
                    'other_images' => $other_images,
                    'files' => $files,
                    'is_pay_later_allowed' => $is_pay_later_allowed,
                    'status' => ($this->request->getPost('status') == "on") ? 1 : 0,
                    'is_cancelable' => $is_cancelable,
                    'cancelable_till' => ($is_cancelable == "1") ? $this->request->getVar('cancelable_till') : '',
                    'at_store' => ($this->request->getPost('at_store') == "on") ? 1 : 0,
                    'at_doorstep' => ($this->request->getPost('at_doorstep') == "on") ? 1 : 0,
                    'approved_by_admin' => $this->request->getPost('approve_service_value'),
                    // Store default language data in main table
                    'title' => $defaultTitle,
                    'description' => $defaultDescription,
                    'long_description' => $defaultLongDescription,
                    'tags' => $defaultTags,
                    'faqs' => $defaultFaqs,
                ];

                if ($this->service->update($Service_id, $service)) {
                    // Handle translated fields using ServiceService
                    $postData = $this->request->getPost();

                    // LOG: Start of translation processing
                    // log_message('debug', '[TRANSLATION_DEBUG] Starting translation processing for service ID: ' . $Service_id);
                    // log_message('debug', '[TRANSLATION_DEBUG] POST data received: ' . json_encode($postData, JSON_UNESCAPED_UNICODE));

                    // Get existing translations for the service
                    $existingTranslationsResult = $this->serviceService->getServiceWithTranslations($Service_id);
                    $existingTranslations = $existingTranslationsResult['translated_data'] ?? [];

                    // LOG: Existing translations
                    // log_message('debug', '[TRANSLATION_DEBUG] Existing translations for service ' . $Service_id . ': ' . json_encode($existingTranslations, JSON_UNESCAPED_UNICODE));

                    // Transform form data to translated_fields structure
                    $translatedFields = $this->transformFormDataToTranslatedFields($postData, $defaultLanguage, $Service_id, $existingTranslations);

                    // LOG: Transformed fields
                    // log_message('debug', '[TRANSLATION_DEBUG] Transformed fields: ' . json_encode($translatedFields, JSON_UNESCAPED_UNICODE));

                    // Add translated_fields to postData for ServiceService
                    $postData['translated_fields'] = $translatedFields;

                    $translationResult = $this->serviceService->handleServiceUpdateWithTranslations($postData, $service, $Service_id, $defaultLanguage);

                    // LOG: Translation result
                    // log_message('debug', '[TRANSLATION_DEBUG] Translation result: ' . json_encode($translationResult, JSON_UNESCAPED_UNICODE));

                    // if (!$translationResult['success']) {
                    //     // If translation saving fails, we should handle this appropriately
                    //     // For now, we'll log the error but continue with the process
                    //     log_message('error', '[TRANSLATION_ERROR] Failed to save service translations: ' . implode(', ', $translationResult['errors']));
                    // } else {
                    //     log_message('debug', '[TRANSLATION_DEBUG] Successfully processed translations for service ' . $Service_id);
                    // }

                    try {
                        $this->saveServiceSeoSettings($Service_id); // Save SEO settings
                    } catch (\Throwable $th) {
                        log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - update_service() - SEO settings');
                        return ErrorResponse("Failed to save SEO settings: " . $th->getMessage(), true, [], [], 200, csrf_token(), csrf_hash());
                    }
                    return successResponse("Service updated successfully", false, [], [], 200, csrf_token(), csrf_hash());
                } else {
                    return ErrorResponse("Service cannot be saved!", true, [], [], 200, csrf_token(), csrf_hash());
                }
            } else {
                return redirect()->to('partner/services');
            }
        } catch (\Throwable $th) {

            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - update_service()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function edit_service()
    {
        try {
            helper('function');
            $uri = service('uri');

            if ($this->isLoggedIn && $this->userIsAdmin) {
                $disk = fetch_current_file_manager();

                $service_id = $uri->getSegments()[3];

                setPageInfo($this->data, labels('services', 'Services') . ' | ' . labels('admin_panel', 'Admin Panel'), 'services');
                // $this->data['categories_name'] = fetch_details('categories', [], ['id', 'name', 'parent_id']);
                $this->data['categories_name'] = get_categories_with_translated_names();

                $service = fetch_details('services', ['id' => $service_id])[0];

                // Debug: Log the raw service data from database (uncomment for debugging)
                // log_message('debug', 'Raw service FAQs from database: ' . ($service['faqs'] ?? 'null'));
                if ($disk == 'local_server') {
                    $localPath = base_url($service['image']);
                    if (check_exists($localPath)) {
                        $service['image'] = $localPath;
                    } else {
                        $service['image'] = '';
                    }
                } else if ($disk == "aws_s3") {
                    $service['image'] = fetch_cloud_front_url('services', $service['image']);
                } else {
                    $service['image'] = $service['image'];
                }
                if (!empty($service['other_images'])) {
                    $service['other_images'] = array_map(function ($data) use ($service, $disk) {
                        if ($disk === "local_server") {
                            return base_url($data);
                        } elseif ($disk === "aws_s3") {
                            return fetch_cloud_front_url('services', $data);
                        }
                    }, json_decode($service['other_images'], true));
                } else {
                    $service['other_images'] = [];
                }
                if (!empty($service['files'])) {
                    $service['files'] = array_map(function ($data) use ($service, $disk) {
                        if ($disk === "local_server") {
                            return base_url($data);
                        } elseif ($disk === "aws_s3") {
                            return fetch_cloud_front_url('services', $data);
                        }
                    }, json_decode($service['files'], true));
                } else {
                    $service['files'] = [];
                }

                // Process FAQs data - decode JSON string to array for proper handling in view
                if (!empty($service['faqs'])) {
                    $faqsData = json_decode($service['faqs'], true);
                    if (is_array($faqsData)) {
                        $faqs = [];
                        // Handle both old array format [["question","answer"]] and new object format [{"question":"q","answer":"a"}]
                        if (isset($faqsData[0])) {
                            if (is_array($faqsData[0]) && count($faqsData[0]) >= 2 && !isset($faqsData[0]['question'])) {
                                // Old array format - direct array of pairs [["question","answer"]]
                                foreach ($faqsData as $pair) {
                                    if (is_array($pair) && count($pair) >= 2) {
                                        $faq = [
                                            'question' => $pair[0],
                                            'answer' => $pair[1]
                                        ];
                                        $faqs[] = $faq;
                                    }
                                }
                            } elseif (is_array($faqsData[0]) && isset($faqsData[0]['question']) && isset($faqsData[0]['answer'])) {
                                // New object format - array of objects [{"question":"q","answer":"a"}]
                                $faqs = $faqsData;
                            } else {
                                // Object format - object with numeric keys {"1":["question","answer"]}
                                foreach ($faqsData as $key => $pair) {
                                    if (is_array($pair) && count($pair) >= 2) {
                                        $faq = [
                                            'question' => $pair[0],
                                            'answer' => $pair[1]
                                        ];
                                        $faqs[] = $faq;
                                    }
                                }
                            }
                        }
                        $service['faqs'] = $faqs;
                    } else {
                        $service['faqs'] = [];
                    }
                } else {
                    $service['faqs'] = [];
                }

                // Debug: Log the processed FAQ data (uncomment for debugging)
                // log_message('debug', 'Processed service FAQs: ' . json_encode($service['faqs']));

                $this->data['service'] = $service;

                // Get current language for translation
                $currentLanguage = get_current_language();

                // Fetch partner data with translated company names based on current language
                $partner_data = $this->db->table('users u')
                    ->select('u.id,u.username,pd.company_name,pd.at_store,pd.at_doorstep,pd.need_approval_for_the_service,tpd.company_name as translated_company_name')
                    ->join('partner_details pd', 'pd.partner_id = u.id')
                    ->join('translated_partner_details tpd', 'tpd.partner_id = pd.partner_id AND tpd.language_code = "' . $currentLanguage . '"', 'left')
                    ->where('pd.is_approved', '1')
                    ->get()->getResultArray();

                // Process partner data to use translated company names when available
                foreach ($partner_data as &$partner) {
                    // Use translated company name if available and not empty, otherwise use original
                    if (!empty($partner['translated_company_name'])) {
                        $partner['display_company_name'] = $partner['translated_company_name'];
                    } else {
                        $partner['display_company_name'] = $partner['company_name'];
                    }
                }

                $this->data['partner_name'] = $partner_data;
                // Fetch taxes with translated names based on current language
                $tax_data = get_taxes_with_translated_names(['status' => 1], ['id', 'title', 'percentage']);
                $this->data['tax_data'] = $tax_data;
                $this->data['main_page'] = 'edit_service';

                $this->seoModel->setTableContext('services');
                $seo_settings = $this->seoModel->getSeoSettingsByReferenceId($service_id, 'full');
                $this->data['service_seo_settings'] = $seo_settings;

                // fetch languages
                $languages = fetch_details('languages', [], ['id', 'language', 'is_default', 'code'], "", '0', 'id', 'ACE');
                $this->data['languages'] = $languages;

                // Load translated service details using ServiceService
                $translatedData = $this->serviceService->getServiceWithTranslations($service_id);

                // Process FAQ data with proper fallback logic
                // For each language, try to get FAQs from translations table first, then fall back to main table
                $mergedServiceDetails = $service;

                foreach ($languages as $language) {
                    $languageCode = $language['code'];
                    $isDefaultLanguage = $language['is_default'] == 1;

                    // Initialize FAQ data for this language
                    $languageFaqs = [];

                    // First, try to get FAQs from translations table
                    if ($translatedData['success'] && isset($translatedData['translated_data'][$languageCode])) {
                        $translation = $translatedData['translated_data'][$languageCode];

                        if (!empty($translation['faqs'])) {
                            // Process translated FAQs data - decode JSON string to array if needed
                            if (is_array($translation['faqs'])) {
                                $languageFaqs = $translation['faqs'];
                            } else {
                                $translatedFaqsData = json_decode($translation['faqs'], true);
                                if (is_array($translatedFaqsData)) {
                                    // Handle both old array format [["question","answer"]] and new object format [{"question":"q","answer":"a"}]
                                    if (isset($translatedFaqsData[0])) {
                                        if (is_array($translatedFaqsData[0]) && count($translatedFaqsData[0]) >= 2 && !isset($translatedFaqsData[0]['question'])) {
                                            // Old array format - direct array of pairs [["question","answer"]]
                                            foreach ($translatedFaqsData as $pair) {
                                                if (is_array($pair) && count($pair) >= 2) {
                                                    $faq = [
                                                        'question' => $pair[0],
                                                        'answer' => $pair[1]
                                                    ];
                                                    $languageFaqs[] = $faq;
                                                }
                                            }
                                        } elseif (is_array($translatedFaqsData[0]) && isset($translatedFaqsData[0]['question']) && isset($translatedFaqsData[0]['answer'])) {
                                            // New object format - array of objects [{"question":"q","answer":"a"}]
                                            $languageFaqs = $translatedFaqsData;
                                        } else {
                                            // Object format - object with numeric keys {"1":["question","answer"]}
                                            foreach ($translatedFaqsData as $key => $pair) {
                                                if (is_array($pair) && count($pair) >= 2) {
                                                    $faq = [
                                                        'question' => $pair[0],
                                                        'answer' => $pair[1]
                                                    ];
                                                    $languageFaqs[] = $faq;
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // If no FAQs found in translations table, fall back to main table (for default language)
                    if (empty($languageFaqs) && $isDefaultLanguage) {
                        if (isset($service['faqs']) && is_array($service['faqs']) && !empty($service['faqs'])) {
                            $languageFaqs = $service['faqs'];
                            // log_message('debug', 'Using FAQs from main table for default language: ' . json_encode($languageFaqs));
                        }
                    }

                    // Set FAQs for this language
                    if ($isDefaultLanguage) {
                        // For default language, set directly in main service data
                        $mergedServiceDetails['faqs'] = $languageFaqs;
                        // log_message('debug', 'Set default language FAQs: ' . json_encode($languageFaqs));
                    } else {
                        // For other languages, set in translated data
                        if (!isset($mergedServiceDetails['translated_' . $languageCode])) {
                            $mergedServiceDetails['translated_' . $languageCode] = [];
                        }
                        $mergedServiceDetails['translated_' . $languageCode]['faqs'] = $languageFaqs;
                        // log_message('debug', 'Set translated FAQs for ' . $languageCode . ': ' . json_encode($languageFaqs));
                    }

                    // Set other translated fields with proper fallback logic
                    if ($translatedData['success'] && isset($translatedData['translated_data'][$languageCode])) {
                        $translation = $translatedData['translated_data'][$languageCode];

                        // Process other translatable fields with fallback logic
                        $translatedTitle = !empty($translation['title']) ? $translation['title'] : ($isDefaultLanguage ? $service['title'] : '');
                        $translatedDescription = !empty($translation['description']) ? $translation['description'] : ($isDefaultLanguage ? $service['description'] : '');
                        $translatedLongDescription = !empty($translation['long_description']) ? $translation['long_description'] : ($isDefaultLanguage ? $service['long_description'] : '');
                        $translatedTags = !empty($translation['tags']) ? $translation['tags'] : ($isDefaultLanguage ? $service['tags'] : '');

                        if (!$isDefaultLanguage) {
                            $mergedServiceDetails['translated_' . $languageCode] = [
                                'title' => $translatedTitle,
                                'description' => $translatedDescription,
                                'long_description' => $translatedLongDescription,
                                'tags' => $translatedTags,
                                'faqs' => $languageFaqs
                            ];
                        } else {
                            // For default language, update main service data with translated data if available
                            if (!empty($translation['title'])) {
                                $mergedServiceDetails['title'] = $translation['title'];
                            }
                            if (!empty($translation['description'])) {
                                $mergedServiceDetails['description'] = $translation['description'];
                            }
                            if (!empty($translation['long_description'])) {
                                $mergedServiceDetails['long_description'] = $translation['long_description'];
                            }
                            if (!empty($translation['tags'])) {
                                $mergedServiceDetails['tags'] = $translation['tags'];
                            }
                        }
                    }
                }

                $this->data['service'] = $mergedServiceDetails;

                // Final debug: Log the final service data being passed to view (uncomment for debugging)
                // log_message('debug', 'Final service FAQs being passed to view: ' . json_encode($this->data['service']['faqs'] ?? 'not set'));

                return view('backend/admin/template', $this->data);
            } else {
                return redirect('admin/login');
            }
        } catch (\Throwable $th) {
            throw $th;
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - edit_service()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function add_service_view()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $permission = is_permitted($this->creator_id, 'create', 'services');
            if (!$permission) {
                return NoPermission();
            }
            setPageInfo($this->data, labels('add_service', 'Add Service') . ' | ' . labels('admin_panel', 'Admin Panel'), 'add_service');
            $partner_details = !empty(fetch_details('partner_details', ['partner_id' => $this->userId])) ? fetch_details('partner_details', ['partner_id' => $this->userId])[0] : [];
            $partner_timings = !empty(fetch_details('partner_timings', ['partner_id' => $this->userId])) ? fetch_details('partner_timings', ['partner_id' => $this->userId]) : [];
            $this->data['data'] = fetch_details('users', ['id' => $this->userId])[0];
            $currency = get_settings('general_settings', true);
            if (empty($currency)) {
                $_SESSION['toastMessage'] = labels('Please first add currency and basic details in general settings', 'Please first add currency and basic details in general settings');
                $_SESSION['toastMessageType'] = 'error';
                $this->session->markAsFlashdata('toastMessage');
                $this->session->markAsFlashdata('toastMessageType');
                return redirect()->to('admin/settings/general-settings')->withCookies();
            }
            $this->data['currency'] = $currency['currency'];
            $this->data['partner_details'] = $partner_details;
            $this->data['partner_timings'] = $partner_timings;
            $this->data['city_name'] = fetch_details('cities', [], ['id', 'name']);
            // $this->data['categories_name'] = fetch_details('categories', [], ['id', 'name', 'parent_id']);
            $this->data['categories_name'] = get_categories_with_translated_names();
            // $this->data['categories_tree'] = $this->getCategoriesTree();


            // Get current language for translation
            $currentLanguage = get_current_language();
            $defaultLanguage = get_default_language();

            // Fetch partner data with translated company names based on current language
            $partner_data = $this->db->table('users u')
                ->select('u.id,u.username,pd.company_name,pd.number_of_members,pd.at_store,pd.at_doorstep,pd.need_approval_for_the_service,tpd.company_name as translated_company_name')
                ->join('partner_details pd', 'pd.partner_id = u.id')
                ->join('translated_partner_details tpd', 'tpd.partner_id = pd.partner_id AND tpd.language_code = "' . $currentLanguage . '"', 'left')
                ->where('pd.is_approved', '1')
                ->get()->getResultArray();

            // Process partner data to use translated company names when available
            foreach ($partner_data as &$partner) {
                // Use translated company name if available and not empty, otherwise use original
                if (!empty($partner['translated_company_name'])) {
                    $partner['display_company_name'] = $partner['translated_company_name'];
                } else {
                    $partner['display_company_name'] = $partner['company_name'];
                }
            }

            $this->data['partner_name'] = $partner_data;
            // Fetch taxes with translated names based on current language
            $tax_data = get_taxes_with_translated_names(['status' => 1], ['id', 'title', 'percentage']);
            $this->data['tax_data'] = $tax_data;

            // fetch languages
            $languages = fetch_details('languages', [], ['id', 'language', 'is_default', 'code'], "", '0', 'id', 'ACE');
            $this->data['languages'] = $languages;

            return view('backend/admin/template', $this->data);
        } catch (\Throwable $th) {
            throw $th;
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - add_service_view()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function service_detail()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return redirect('admin/login');
            }
            $uri = service('uri');
            $service_id = $uri->getSegments()[3];
            setPageInfo($this->data, labels('services', 'Services') . ' | ' . labels('admin_panel', 'Admin Panel'), 'service_details');
            // $this->data['categories_name'] = fetch_details('categories', [], ['id', 'name']);
            $this->data['categories_name'] = get_categories_with_translated_names();
            $this->data['categories_tree'] = $this->getCategoriesTree();
            $partner_data = $this->db->table('users u')
                ->select('u.id,u.username,pd.company_name,pd.number_of_members')
                ->join('partner_details pd', 'pd.partner_id = u.id')
                ->where('is_approved', '1')
                ->get()->getResultArray();
            $this->data['partner_name'] = $partner_data;
            // Fetch taxes with translated names based on current language
            $tax_data = get_taxes_with_translated_names(['status' => 1], ['id', 'title', 'percentage']);
            $service = fetch_details('services', ['id' => $service_id]);
            $disk = fetch_current_file_manager();

            if (!empty($service)) {
                if ($disk == 'local_server') {
                    $localPath = base_url('/public/uploads/services/' . $service[0]['image']);
                    if (check_exists($localPath)) {
                        $service[0]['image'] = $localPath;
                    } else {
                        $service[0]['image'] = '';
                    }
                } else if ($disk == "aws_s3") {
                    $service[0]['image'] = fetch_cloud_front_url('services', $service[0]['image']);
                } else {
                    $service[0]['image'] = $service[0]['image'];
                }
                if (!empty($service[0]['other_images'])) {
                    $decodedOtherImages = json_decode($service[0]['other_images'], true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $service[0]['other_images'] = array_map(function ($data) use ($service, $disk) {
                            if ($disk === "local_server") {
                                return base_url($data);
                            } elseif ($disk === "aws_s3") {
                                return fetch_cloud_front_url('services', $data);
                            }
                            return $data;
                        }, $decodedOtherImages);
                    } else {
                        $service[0]['other_images'] = [];
                    }
                } else {
                    $service[0]['other_images'] = [];
                }
                if (!empty($service[0]['files'])) {
                    $decodedFiles = json_decode($service[0]['files'], true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $service[0]['files'] = array_map(function ($data) use ($service, $disk) {
                            if ($disk === "local_server") {
                                return base_url($data);
                            } elseif ($disk === "aws_s3") {
                                return fetch_cloud_front_url('services', $data);
                            }
                            return $data;
                        }, $decodedFiles);
                    } else {
                        $service[0]['files'] = [];
                    }
                } else {
                    $service[0]['files'] = [];
                }
            } else {
                return redirect('admin/login');
            }
            $this->data['service'] = $service;
            $this->data['tax_data'] = $tax_data;
            return view('backend/admin/template', $this->data);
        } catch (\Throwable $th) {

            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - service_detail()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function disapprove_service()
    {
        try {
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }
            $permission = is_permitted($this->creator_id, 'update', 'services');
            if ($permission) {
                if ($this->isLoggedIn && $this->userIsAdmin) {
                    $partner_id = $this->request->getPost('partner_id');
                    $service_id = $this->request->getPost('service_id');
                    $builder = $this->db->table('services');
                    $service_approval = $builder->set('approved_by_admin', 0)->where('user_id', $partner_id)->where('id', $service_id)->update();

                    $to_send_id = $partner_id;

                    $usersEmail = fetch_details('users', ['id' => $to_send_id], ['email']);
                    $users_fcm = $this->db->table('users_fcm_ids')
                        ->select('fcm_id,platform')
                        ->where('user_id', $to_send_id)
                        ->whereIn('platform', ['android', 'ios', 'provider_panel'])
                        ->where('status', '1')
                        ->get()
                        ->getResultArray();
                    foreach ($users_fcm as $ids) {
                        if ($ids['fcm_id'] != "") {
                            $fcm_ids['fcm_id'] = $ids['fcm_id'];
                            $fcm_ids['platform'] = $ids['platform'];
                        }
                    }
                    if (!empty($fcm_ids) && check_notification_setting('service_disapproved', 'notification')) {

                        $title = $this->trans->serviceRequestRejection;
                        $msg = $this->trans->serviceRequestApprovalRejectedMessage;
                        $type = 'service_request_status';
                        $fcmMsg = array(
                            'content_available' => "true",
                            'title' => $title,
                            'body' => $msg,
                            'type_id' => "$to_send_id",
                            'type' => $type,
                            'status' => 'reject',
                            'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                        );
                        $registrationIDs_chunks = array_chunk($users_fcm, 1000);
                        // Use queued notifications for better performance
                        queue_notification($fcmMsg, $registrationIDs_chunks, [], 'default');


                        $store_notification = store_notifications(
                            $title,
                            $msg,
                            $type,
                            $to_send_id,
                            0,
                            'general',
                            now(),
                            'specific_user',
                            '',
                            '',
                            $to_send_id,
                            '',
                            '',
                            '',
                            '',

                        );
                    }
                    if ($service_approval) {
                        if (!empty($usersEmail[0]['email']) && check_notification_setting('service_disapproved', 'email') && is_unsubscribe_enabled($partner_id) == 1) {
                            send_custom_email('service_disapproved', $partner_id, $usersEmail[0]['email']);
                        }
                        if (check_notification_setting('service_disapproved', 'sms')) {
                            send_custom_sms('service_disapproved', $partner_id, $usersEmail[0]['email']);
                        }
                        return successResponse("Service is disapproved", false, [], [], 200, csrf_token(), csrf_hash());
                    } else {
                        return successResponse("Could not disapprove service", false, [$service_approval], [], 200, csrf_token(), csrf_hash());
                    }
                } else {
                    return redirect('admin/login');
                }
            } else {
                return NoPermission();
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - disapprove_service()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function approve_service()
    {
        try {
            $result = checkModificationInDemoMode($this->superadmin);
            if ($result !== true) {
                return $this->response->setJSON($result);
            }
            $permission = is_permitted($this->creator_id, 'update', 'services');
            if (!$permission) {
                return NoPermission();
            }
            $partner_id = $this->request->getPost('partner_id');
            $service_id = $this->request->getPost('service_id');
            $builder = $this->db->table('services');
            $service_approval = $builder->set('approved_by_admin', 1)->where('user_id', $partner_id)->where('id', $service_id)->update();

            $to_send_id = $partner_id;

            $usersEmail = fetch_details('users', ['id' => $to_send_id], ['email']);
            $users_fcm = $this->db->table('users_fcm_ids')
                ->select('fcm_id,platform')
                ->where('user_id', $to_send_id)
                ->whereIn('platform', ['android', 'ios', 'provider_panel'])
                ->where('status', '1')
                ->get()
                ->getResultArray();
            foreach ($users_fcm as $ids) {
                if ($ids['fcm_id'] != "") {
                    $fcm_ids['fcm_id'] = $ids['fcm_id'];
                    $fcm_ids['platform'] = $ids['platform'];
                }
            }
            if (!empty($fcm_ids) && check_notification_setting('service_approved', 'notification')) {

                $title = $this->trans->serviceRequestApproval;
                $msg = $this->trans->serviceApprovalRequestApprovedMessage;
                $type = 'service_request_status';
                $fcmMsg = array(
                    'content_available' => "true",
                    'title' => $title,
                    'body' => $msg,
                    'type' => $type,
                    'status' => 'approve',
                    'type_id' => "$to_send_id",
                    'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                );
                $registrationIDs_chunks = array_chunk($users_fcm, 1000);
                // Use queued notifications for better performance
                queue_notification($fcmMsg, $registrationIDs_chunks, [], 'default');
                $store_notification = store_notifications($title, $msg, $type, $to_send_id, 0, 'general', now(), 'specific_user', '', '', $to_send_id, '', '', '', '', '');
            }
            if ($service_approval) {
                if (!empty($usersEmail[0]['email']) && check_notification_setting('service_approved', 'email') && is_unsubscribe_enabled($partner_id) == 1) {
                    send_custom_email('service_approved', $partner_id, $usersEmail[0]['email']);
                }
                if (check_notification_setting('service_approved', 'sms')) {
                    send_custom_sms('service_approved', $partner_id, $usersEmail[0]['email']);
                }
                return successResponse("Service is approved", false, [], [], 200, csrf_token(), csrf_hash());
            } else {
                return successResponse("Could not Approve service", false, [$service_approval], [], 200, csrf_token(), csrf_hash());
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - approve_service()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function duplicate()
    {
        try {
            helper('function');
            $uri = service('uri');
            if ($this->isLoggedIn && $this->userIsAdmin) {
                $service_id = $uri->getSegments()[3];
                setPageInfo($this->data, labels('services', 'Services') . ' | ' . labels('admin_panel', 'Admin Panel'), 'services');
                $this->data['categories_name'] = fetch_details('categories', [], ['id', 'name', 'parent_id']);

                // Fetch the original service data
                $original_service = fetch_details('services', ['id' => $service_id])[0];
                $disk = fetch_current_file_manager();

                // Handle main image
                if (!empty($original_service['image'])) {
                    if ($disk == 'local_server') {
                        $original_service['image'] = base_url($original_service['image']);
                    } else if ($disk == "aws_s3") {
                        $original_service['image'] = fetch_cloud_front_url('services', $original_service['image']);
                    }
                }

                // Handle other images
                if (!empty($original_service['other_images'])) {
                    $other_images = json_decode($original_service['other_images'], true);
                    if (is_array($other_images)) {
                        $original_service['other_images'] = array_map(function ($data) use ($disk) {
                            if ($disk === "local_server") {
                                return base_url($data);
                            } elseif ($disk === "aws_s3") {
                                return fetch_cloud_front_url('services', $data);
                            } else {
                                return $data;
                            }
                        }, $other_images);
                    } else {
                        $original_service['other_images'] = [];
                    }
                } else {
                    $original_service['other_images'] = [];
                }

                // Handle files
                if (!empty($original_service['files'])) {
                    $files = json_decode($original_service['files'], true);
                    if (is_array($files)) {
                        $original_service['files'] = array_map(function ($data) use ($disk) {
                            if ($disk === "local_server") {
                                return base_url($data);
                            } elseif ($disk === "aws_s3") {
                                return fetch_cloud_front_url('services', $data);
                            } else {
                                return $data;
                            }
                        }, $files);
                    } else {
                        $original_service['files'] = [];
                    }
                } else {
                    $original_service['files'] = [];
                }

                // Process FAQs data - decode JSON string to array for proper handling in view
                if (!empty($original_service['faqs'])) {
                    $faqsData = json_decode($original_service['faqs'], true);
                    if (is_array($faqsData)) {
                        $faqs = [];
                        // Handle both old array format [["question","answer"]] and new object format [{"question":"q","answer":"a"}]
                        if (isset($faqsData[0])) {
                            if (is_array($faqsData[0]) && count($faqsData[0]) >= 2 && !isset($faqsData[0]['question'])) {
                                // Old array format - direct array of pairs [["question","answer"]]
                                foreach ($faqsData as $pair) {
                                    if (is_array($pair) && count($pair) >= 2) {
                                        $faq = [
                                            'question' => $pair[0],
                                            'answer' => $pair[1]
                                        ];
                                        $faqs[] = $faq;
                                    }
                                }
                            } elseif (is_array($faqsData[0]) && isset($faqsData[0]['question']) && isset($faqsData[0]['answer'])) {
                                // New object format - array of objects [{"question":"q","answer":"a"}]
                                $faqs = $faqsData;
                            } else {
                                // Object format - object with numeric keys {"1":["question","answer"]}
                                foreach ($faqsData as $key => $pair) {
                                    if (is_array($pair) && count($pair) >= 2) {
                                        $faq = [
                                            'question' => $pair[0],
                                            'answer' => $pair[1]
                                        ];
                                        $faqs[] = $faq;
                                    }
                                }
                            }
                        }
                        $original_service['faqs'] = $faqs;
                    } else {
                        $original_service['faqs'] = [];
                    }
                } else {
                    $original_service['faqs'] = [];
                }

                $this->data['service'] = $original_service;

                // Get current language for translation
                $currentLanguage = get_current_language();

                // Fetch partner data with translated company names based on current language
                $partner_data = $this->db->table('users u')
                    ->select('u.id,u.username,pd.company_name,pd.at_store,pd.at_doorstep,pd.need_approval_for_the_service,tpd.company_name as translated_company_name')
                    ->join('partner_details pd', 'pd.partner_id = u.id')
                    ->join('translated_partner_details tpd', 'tpd.partner_id = pd.partner_id AND tpd.language_code = "' . $currentLanguage . '"', 'left')
                    ->where('pd.is_approved', '1')
                    ->get()->getResultArray();

                // Process partner data to use translated company names when available
                foreach ($partner_data as &$partner) {
                    // Use translated company name if available and not empty, otherwise use original
                    if (!empty($partner['translated_company_name'])) {
                        $partner['display_company_name'] = $partner['translated_company_name'];
                    } else {
                        $partner['display_company_name'] = $partner['company_name'];
                    }
                }

                $this->data['partner_name'] = $partner_data;
                // Fetch taxes with translated names based on current language
                $tax_data = get_taxes_with_translated_names(['status' => 1], ['id', 'title', 'percentage']);
                $this->data['tax_data'] = $tax_data;
                $this->seoModel->setTableContext('services');
                $seo_settings = $this->seoModel->getSeoSettingsByReferenceId($service_id, 'full');
                $this->data['service_seo_settings'] = $seo_settings;
                $this->data['main_page'] = 'service_clone';

                // fetch languages
                $languages = fetch_details('languages', [], ['id', 'language', 'is_default', 'code'], "", '0', 'id', 'ACE');
                $this->data['languages'] = $languages;

                // Load translated service details using ServiceService
                $translatedData = $this->serviceService->getServiceWithTranslations($service_id);

                // Process FAQ data with proper fallback logic
                // For each language, try to get FAQs from translations table first, then fall back to main table
                $mergedServiceDetails = $original_service;

                foreach ($languages as $language) {
                    $languageCode = $language['code'];
                    $isDefaultLanguage = $language['is_default'] == 1;

                    // Initialize FAQ data for this language
                    $languageFaqs = [];

                    // First, try to get FAQs from translations table
                    if ($translatedData['success'] && isset($translatedData['translated_data'][$languageCode])) {
                        $translation = $translatedData['translated_data'][$languageCode];

                        if (!empty($translation['faqs'])) {
                            // Process translated FAQs data - decode JSON string to array if needed
                            if (is_array($translation['faqs'])) {
                                $languageFaqs = $translation['faqs'];
                            } else {
                                $translatedFaqsData = json_decode($translation['faqs'], true);
                                if (is_array($translatedFaqsData)) {
                                    // Handle both old array format [["question","answer"]] and new object format [{"question":"q","answer":"a"}]
                                    if (isset($translatedFaqsData[0])) {
                                        if (is_array($translatedFaqsData[0]) && count($translatedFaqsData[0]) >= 2 && !isset($translatedFaqsData[0]['question'])) {
                                            // Old array format - direct array of pairs [["question","answer"]]
                                            foreach ($translatedFaqsData as $pair) {
                                                if (is_array($pair) && count($pair) >= 2) {
                                                    $faq = [
                                                        'question' => $pair[0],
                                                        'answer' => $pair[1]
                                                    ];
                                                    $languageFaqs[] = $faq;
                                                }
                                            }
                                        } elseif (is_array($translatedFaqsData[0]) && isset($translatedFaqsData[0]['question']) && isset($translatedFaqsData[0]['answer'])) {
                                            // New object format - array of objects [{"question":"q","answer":"a"}]
                                            $languageFaqs = $translatedFaqsData;
                                        } else {
                                            // Object format - object with numeric keys {"1":["question","answer"]}
                                            foreach ($translatedFaqsData as $key => $pair) {
                                                if (is_array($pair) && count($pair) >= 2) {
                                                    $faq = [
                                                        'question' => $pair[0],
                                                        'answer' => $pair[1]
                                                    ];
                                                    $languageFaqs[] = $faq;
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // If no FAQs found in translations table, fall back to main table (for default language)
                    if (empty($languageFaqs) && $isDefaultLanguage) {
                        if (isset($original_service['faqs']) && is_array($original_service['faqs']) && !empty($original_service['faqs'])) {
                            $languageFaqs = $original_service['faqs'];
                        }
                    }

                    // Set FAQs for this language
                    if ($isDefaultLanguage) {
                        // For default language, set directly in main service data
                        $mergedServiceDetails['faqs'] = $languageFaqs;
                    } else {
                        // For other languages, set in translated data
                        if (!isset($mergedServiceDetails['translated_' . $languageCode])) {
                            $mergedServiceDetails['translated_' . $languageCode] = [];
                        }
                        $mergedServiceDetails['translated_' . $languageCode]['faqs'] = $languageFaqs;
                    }

                    // Set other translated fields with proper fallback logic
                    if ($translatedData['success'] && isset($translatedData['translated_data'][$languageCode])) {
                        $translation = $translatedData['translated_data'][$languageCode];

                        // Process other translatable fields with fallback logic
                        $translatedTitle = !empty($translation['title']) ? $translation['title'] : ($isDefaultLanguage ? $original_service['title'] : '');
                        $translatedDescription = !empty($translation['description']) ? $translation['description'] : ($isDefaultLanguage ? $original_service['description'] : '');
                        $translatedLongDescription = !empty($translation['long_description']) ? $translation['long_description'] : ($isDefaultLanguage ? $original_service['long_description'] : '');
                        $translatedTags = !empty($translation['tags']) ? $translation['tags'] : ($isDefaultLanguage ? $original_service['tags'] : '');

                        if (!$isDefaultLanguage) {
                            $mergedServiceDetails['translated_' . $languageCode] = [
                                'title' => $translatedTitle,
                                'description' => $translatedDescription,
                                'long_description' => $translatedLongDescription,
                                'tags' => $translatedTags,
                                'faqs' => $languageFaqs
                            ];
                        } else {
                            // For default language, update main service data with translated data if available
                            if (!empty($translation['title'])) {
                                $mergedServiceDetails['title'] = $translation['title'];
                            }
                            if (!empty($translation['description'])) {
                                $mergedServiceDetails['description'] = $translation['description'];
                            }
                            if (!empty($translation['long_description'])) {
                                $mergedServiceDetails['long_description'] = $translation['long_description'];
                            }
                            if (!empty($translation['tags'])) {
                                $mergedServiceDetails['tags'] = $translation['tags'];
                            }
                        }
                    }
                }

                $this->data['service'] = $mergedServiceDetails;

                return view('backend/admin/template', $this->data);
            } else {
                return redirect('admin/login');
            }
        } catch (\Throwable $th) {
            throw $th;
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - duplicate()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function bulk_import_services()
    {
        if ($this->isLoggedIn && $this->userIsAdmin) {
            setPageInfo($this->data, labels('services', 'Services') . ' | ' . labels('admin_panel', 'Admin Panel'), 'bulk_import_services');
            $partner_data = $this->db->table('users u')
                ->select('u.id,u.username,pd.company_name,pd.number_of_members')
                ->join('partner_details pd', 'pd.partner_id = u.id')
                ->where('is_approved', '1')
                ->get()->getResultArray();
            $this->data['partner_name'] = $partner_data;
            return view('backend/admin/template', $this->data);
        } else {
            return redirect('admin/login');
        }
    }

    public function bulk_import_service_upload()
    {
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            return $this->response->setJSON($result);
        }
        $file = $this->request->getFile('file');
        $filePath = FCPATH . 'public/uploads/service_bulk_upload/';
        if (!is_dir($filePath)) {
            if (!mkdir($filePath, 0775, true)) {
                return $this->response->setJSON([
                    'status' => 'error',
                    'message' => labels('Failed to create folders', 'Failed to create folders')
                ]);
            }
        }
        $newName = $file->getRandomName();
        $file->move($filePath, $newName);
        $fullPath = $filePath . $newName;
        $spreadsheet = IOFactory::load($fullPath);
        $sheet = $spreadsheet->getActiveSheet();
        $headerRow = $sheet->getRowIterator(1)->current();
        $cellIterator = $headerRow->getCellIterator();
        $cellIterator->setIterateOnlyExistingCells(false);

        // Parse multilanguage headers for translatable fields
        // Translatable fields: title, description, long_description, tags, faqs
        $languageHeaders = [];
        $other_image_Headers = [];
        $FilesHeaders = [];
        $columnIndex = 0;
        $OtherImagecolumnIndex = 0;
        $FilescolumnIndex = 0;

        $headerRow = $sheet->rangeToArray('A1:' . $sheet->getHighestColumn() . '1', NULL, TRUE, TRUE, TRUE);
        $headers = explode(',', $headerRow[1]['A']);
        $headers = array_map(function ($header) {
            return trim($header, ' "');
        }, $headers);

        // Get available languages from database for validation
        $languages = fetch_details('languages', [], ['id', 'language', 'code', 'is_default'], "", '0', 'id', 'ASC');
        $defaultLanguage = 'en'; // fallback
        foreach ($languages as $language) {
            if ($language['is_default'] == 1) {
                $defaultLanguage = $language['code'];
                break;
            }
        }

        if (!in_array('ID', $headers)) {
            //insert
            // Parse headers for translatable fields with language codes
            // Format: Title[en], Description[es], faq[en][question][1], etc.
            foreach ($cellIterator as $cell) {
                $header = $cell->getValue();

                // Match multilanguage translatable fields: Title[en], Description[en], etc.
                if (preg_match('/^(Title|Description|Long Description|Tags)\[([a-z]{2,})\]$/i', $header, $matches)) {
                    $fieldName = strtolower(str_replace(' ', '_', $matches[1]));
                    $langCode = $matches[2];
                    if (!isset($languageHeaders[$fieldName])) {
                        $languageHeaders[$fieldName] = [];
                    }
                    $languageHeaders[$fieldName][$langCode] = $columnIndex;
                }
                // Match multilanguage FAQs: faq[en][question][1], faq[es][answer][1]
                elseif (preg_match('/^faq\[([a-z]{2,})\]\[(question|answer)\]\[(\d+)\]$/i', $header, $matches)) {
                    $langCode = $matches[1];
                    $type = $matches[2];
                    $faqNumber = $matches[3];
                    if (!isset($languageHeaders['faqs'])) {
                        $languageHeaders['faqs'] = [];
                    }
                    if (!isset($languageHeaders['faqs'][$langCode])) {
                        $languageHeaders['faqs'][$langCode] = [];
                    }
                    if (!isset($languageHeaders['faqs'][$langCode][$faqNumber])) {
                        $languageHeaders['faqs'][$langCode][$faqNumber] = [];
                    }
                    $languageHeaders['faqs'][$langCode][$faqNumber][$type] = $columnIndex;
                }
                // Match other images
                elseif (preg_match('/^Other Image\[(\d+)\]$/', $header, $matches)) {
                    $other_image_number = $matches[1];
                    $other_image_Headers[$other_image_number] = $OtherImagecolumnIndex;
                }
                // Match files
                elseif (preg_match('/^Files\[(\d+)\]$/', $header, $matches)) {
                    $fileNumber = $matches[1];
                    $FilesHeaders[$fileNumber] = $FilescolumnIndex;
                }

                $columnIndex++;
                $OtherImagecolumnIndex++;
                $FilescolumnIndex++;
            }
            $data = $sheet->toArray();
            array_shift($data);
            $data = array_filter($data, function ($row) {
                return !empty(array_filter($row));
            });

            // Validate default language has required fields
            $requiredFields = ['title', 'description'];
            foreach ($requiredFields as $field) {
                if (!isset($languageHeaders[$field][$defaultLanguage])) {
                    return ErrorResponse(
                        labels('MULTILANGUAGE_MISSING_REQUIRED_FIELD', ucfirst($field) . " is required for default language ($defaultLanguage) in CSV headers"),
                        true,
                        [],
                        [],
                        200,
                        csrf_token(),
                        csrf_hash()
                    );
                }
            }

            $services = [];
            $serviceTranslations = [];

            foreach ($data as $rowIndex => $row) {
                // NEW CSV Structure (after multilanguage support):
                // Index 0: Provider ID
                // Index 1: Category ID
                // Index 2: Duration
                // Index 3: Members Required
                // Index 4: Max Quantity
                // Index 5: Price Type
                // Index 6: Tax ID
                // Index 7: Price
                // Index 8: Discounted Price
                // Index 9: Is Cancelable
                // Index 10: Cancelable before
                // Index 11: Pay Later Allowed
                // Index 12: At Store
                // Index 13: At Doorstep
                // Index 14: Status
                // Index 15: Approve Service
                // Index 16: Image
                // Then: Dynamic language columns...

                // Validate references
                $provider = fetch_details('partner_details', ['partner_id' => $row[0]]);
                if (empty($provider)) {
                    return ErrorResponse(labels('Provider ID', 'Provider ID') . " :: " . $row[0] . " " . labels('not found', 'not found'), true, [], [], 200, csrf_token(), csrf_hash());
                }
                $category = fetch_details('categories', ['id' => $row[1]]);
                if (empty($category)) {
                    return ErrorResponse(labels('Category ID', 'Category ID') . " :: " . $row[1] . " " . labels('not found', 'not found'), true, [], [], 200, csrf_token(), csrf_hash());
                }
                $tax = fetch_details('taxes', ['id' => $row[6]]); // FIXED: was $row[10], now $row[6]
                if (empty($tax)) {
                    return ErrorResponse(labels('Tax ID', 'Tax ID') . " :: " . $row[6] . " " . labels('not found', 'not found'), true, [], [], 200, csrf_token(), csrf_hash());
                }

                // Extract translated fields per language
                $translatedFields = $this->extractTranslatedFieldsFromBulkRow($row, $languageHeaders, $defaultLanguage);

                // Validate default language required fields have values
                if (empty($translatedFields['title'][$defaultLanguage])) {
                    return ErrorResponse(
                        labels('MULTILANGUAGE_MISSING_TITLE', "Title is required for default language ($defaultLanguage) at row " . ($rowIndex + 2)),
                        true,
                        [],
                        [],
                        200,
                        csrf_token(),
                        csrf_hash()
                    );
                }
                if (empty($translatedFields['description'][$defaultLanguage])) {
                    return ErrorResponse(
                        labels('MULTILANGUAGE_MISSING_DESCRIPTION', "Description is required for default language ($defaultLanguage) at row " . ($rowIndex + 2)),
                        true,
                        [],
                        [],
                        200,
                        csrf_token(),
                        csrf_hash()
                    );
                }

                // Process other images
                $other_images = [];
                foreach ($other_image_Headers as $indexes) {
                    $other_image = isset($row[$indexes]) ? trim($row[$indexes]) : '';
                    if (!empty($other_image)) {
                        copy_image($row[$indexes], '/public/uploads/services/');
                        if (!empty($other_image)) {
                            $other_images[] = $other_image;
                        }
                    }
                }

                // Process files
                $files = [];
                foreach ($FilesHeaders as $indexes) {
                    $file = isset($row[$indexes]) ? trim($row[$indexes]) : '';
                    if (!empty($file)) {
                        copy_image($row[$indexes], '/public/uploads/services/');
                        $files[] = $file;
                    }
                }

                // Process main service image (FIXED: was $row[20], now $row[16])
                $image = !empty($row[16]) ? copy_image($row[16], '/public/uploads/services/') : "";

                // Prepare service data INCLUDING default language values for fallback
                // Store default language translatable fields in main table as fallback
                // These same values will also be stored in translated_service_details table
                $serviceData = [
                    'user_id' => $row[0], // Index 0: Provider ID
                    'category_id' => $row[1], // Index 1: Category ID
                    // ✅ Include default language translatable fields as fallback
                    'title' => $translatedFields['title'][$defaultLanguage] ?? '',
                    'description' => $translatedFields['description'][$defaultLanguage] ?? '',
                    'long_description' => $translatedFields['long_description'][$defaultLanguage] ?? '',
                    'tags' => $translatedFields['tags'][$defaultLanguage] ?? '',
                    'faqs' => $translatedFields['faqs'][$defaultLanguage] ?? json_encode([], JSON_UNESCAPED_UNICODE),
                    // Non-translatable fields (FIXED: all indexes corrected)
                    'duration' => $row[2], // FIXED: was $row[5], now $row[2]
                    'number_of_members_required' => $row[3], // FIXED: was $row[6], now $row[3]
                    'max_quantity_allowed' => $row[4], // FIXED: was $row[7], now $row[4]
                    'tax_type' => $row[5], // FIXED: was $row[9], now $row[5]
                    'tax_id' => $row[6], // FIXED: was $row[10], now $row[6]
                    'price' => $row[7], // FIXED: was $row[11], now $row[7]
                    'discounted_price' => $row[8], // FIXED: was $row[12], now $row[8]
                    'is_cancelable' => $row[9], // FIXED: was $row[13], now $row[9]
                    'cancelable_till' => ($row[9] == 1) ? $row[10] : "", // FIXED: indexes corrected
                    'is_pay_later_allowed' => $row[11], // FIXED: was $row[15], now $row[11]
                    'at_store' => $row[12], // FIXED: was $row[16], now $row[12]
                    'at_doorstep' => $row[13], // FIXED: was $row[17], now $row[13]
                    'status' => $row[14], // FIXED: was $row[18], now $row[14]
                    'approved_by_admin' => ($provider[0]['need_approval_for_the_service'] == "1") ? "1" : "0",
                    'other_images' => json_encode($other_images),
                    'image' => $image,
                    'files' => json_encode($files),
                ];

                $services[] = $serviceData;
                $serviceTranslations[] = $translatedFields;
            }

            // Insert services and their translations
            $serviceModel = new Service_model();
            $db = \Config\Database::connect();
            $db->transStart();

            try {
                foreach ($services as $index => $service) {
                    // Insert service with default language values as fallback
                    if (!$serviceModel->insert($service)) {
                        throw new \Exception("Failed to add service at row " . ($index + 2));
                    }

                    $serviceId = $serviceModel->insertID();

                    // Save ALL language translations (including default) in translated_service_details table
                    // This provides:
                    // 1. Main table: Has default language values for quick access and fallback
                    // 2. Translations table: Has ALL languages including default for consistency

                    // Prepare translation data for ServicesService
                    $translationData = $serviceTranslations[$index];

                    // Convert FAQ format for ServicesService if needed
                    if (isset($translationData['faqs_by_number']) && !empty($translationData['faqs_by_number'])) {
                        // ServicesService expects FAQs in the format: faqs_by_number
                        $translationData['faqs'] = $translationData['faqs_by_number'];
                        unset($translationData['faqs_by_number']);
                    }

                    $translationResult = $this->serviceService->saveTranslatedFields(
                        $serviceId,
                        $translationData
                    );

                    if (!$translationResult['success']) {
                        $errors = implode(', ', $translationResult['errors']);
                        throw new \Exception("Failed to save translations for service ID $serviceId: " . $errors);
                    }
                }

                $db->transComplete();

                if ($db->transStatus() === false) {
                    throw new \Exception('Transaction failed');
                }

                return successResponse("Services added successfully", false, [], [], 200, csrf_token(), csrf_hash());
            } catch (\Exception $e) {
                $db->transRollback();
                log_message('error', 'Bulk import service error: ' . $e->getMessage());
                return ErrorResponse($e->getMessage(), true, [], [], 200, csrf_token(), csrf_hash());
            }
        } else {
            //update
            // Parse headers for translatable fields with language codes (same as INSERT)
            // Format: Title[en], Description[es], faq[en][question][1], etc.
            foreach ($cellIterator as $cell) {
                $header = $cell->getValue();

                // Match multilanguage translatable fields: Title[en], Description[en], etc.
                if (preg_match('/^(Title|Description|Long Description|Tags)\[([a-z]{2,})\]$/i', $header, $matches)) {
                    $fieldName = strtolower(str_replace(' ', '_', $matches[1]));
                    $langCode = $matches[2];
                    if (!isset($languageHeaders[$fieldName])) {
                        $languageHeaders[$fieldName] = [];
                    }
                    $languageHeaders[$fieldName][$langCode] = $columnIndex;
                }
                // Match multilanguage FAQs: faq[en][question][1], faq[es][answer][1]
                elseif (preg_match('/^faq\[([a-z]{2,})\]\[(question|answer)\]\[(\d+)\]$/i', $header, $matches)) {
                    $langCode = $matches[1];
                    $type = $matches[2];
                    $faqNumber = $matches[3];
                    if (!isset($languageHeaders['faqs'])) {
                        $languageHeaders['faqs'] = [];
                    }
                    if (!isset($languageHeaders['faqs'][$langCode])) {
                        $languageHeaders['faqs'][$langCode] = [];
                    }
                    if (!isset($languageHeaders['faqs'][$langCode][$faqNumber])) {
                        $languageHeaders['faqs'][$langCode][$faqNumber] = [];
                    }
                    $languageHeaders['faqs'][$langCode][$faqNumber][$type] = $columnIndex;
                }
                // Match other images
                elseif (preg_match('/^Other Image\[(\d+)\]$/', $header, $matches)) {
                    $other_image_number = $matches[1];
                    $other_image_Headers[$other_image_number] = $OtherImagecolumnIndex;
                }
                // Match files
                elseif (preg_match('/^Files\[(\d+)\]$/', $header, $matches)) {
                    $fileNumber = $matches[1];
                    $FilesHeaders[$fileNumber] = $FilescolumnIndex;
                }

                $columnIndex++;
                $OtherImagecolumnIndex++;
                $FilescolumnIndex++;
            }
            $data = $sheet->toArray();
            array_shift($data);
            $data = array_filter($data, function ($row) {
                return !empty(array_filter($row));
            });
            // Validate default language has required fields
            $requiredFields = ['title', 'description'];
            foreach ($requiredFields as $field) {
                if (!isset($languageHeaders[$field][$defaultLanguage])) {
                    return ErrorResponse(
                        labels('MULTILANGUAGE_MISSING_REQUIRED_FIELD', ucfirst($field) . " is required for default language ($defaultLanguage) in CSV headers"),
                        true,
                        [],
                        [],
                        200,
                        csrf_token(),
                        csrf_hash()
                    );
                }
            }

            $services = [];
            $serviceTranslations = [];

            foreach ($data as $rowIndex => $row) {
                // NEW CSV Structure for UPDATE (same as INSERT but with ID at index 0):
                // Index 0: ID (SERVICE ID - required for update)
                // Index 1: Provider ID
                // Index 2: Category ID
                // Index 3: Duration
                // Index 4: Members Required
                // Index 5: Max Quantity
                // Index 6: Price Type
                // Index 7: Tax ID
                // Index 8: Price
                // Index 9: Discounted Price
                // Index 10: Is Cancelable
                // Index 11: Cancelable before
                // Index 12: Pay Later Allowed
                // Index 13: At Store
                // Index 14: At Doorstep
                // Index 15: Status
                // Index 16: Approve Service
                // Index 17: Image
                // Then: Dynamic language columns...

                $fetch_service_data = fetch_details('services', ['id' => $row[0]], ['image', 'other_images', 'files']);
                if (empty($fetch_service_data)) {
                    return ErrorResponse(labels('Service ID', 'Service ID') . " :: " . $row[0] . " " . labels('not found', 'not found'), true, [], [], 200, csrf_token(), csrf_hash());
                }

                $old_other_images = [];
                if (!empty($fetch_service_data)) {
                    $other_images = $fetch_service_data[0]['other_images'];
                    $old_other_images = is_string($other_images) ? json_decode($other_images, true) : $other_images;
                    if (json_last_error() !== JSON_ERROR_NONE) {
                        $old_other_images = [];
                    }
                }

                $old_files = [];
                if (!empty($fetch_service_data)) {
                    $old_files = $fetch_service_data[0]['files'];
                    $old_files = is_string($old_files) ? json_decode($old_files, true) : $old_files;
                    if (json_last_error() !== JSON_ERROR_NONE) {
                        $old_files = [];
                    }
                }

                // Validate references - UPDATED INDEXES
                $provider = fetch_details('partner_details', ['partner_id' => $row[1]]);
                if (empty($provider)) {
                    return ErrorResponse(labels('Provider ID', 'Provider ID') . " :: " . $row[1] . " " . labels('not found', 'not found'), true, [], [], 200, csrf_token(), csrf_hash());
                }
                $category = fetch_details('categories', ['id' => $row[2]]);
                if (empty($category)) {
                    return ErrorResponse(labels('Category ID', 'Category ID') . " :: " . $row[2] . " " . labels('not found', 'not found'), true, [], [], 200, csrf_token(), csrf_hash());
                }
                $tax = fetch_details('taxes', ['id' => $row[7]]); // FIXED: was $row[11], now $row[7]
                if (empty($tax)) {
                    return ErrorResponse(labels('Tax ID', 'Tax ID') . " :: " . $row[7] . " " . labels('not found', 'not found'), true, [], [], 200, csrf_token(), csrf_hash());
                }

                // Extract translated fields per language
                $translatedFields = $this->extractTranslatedFieldsFromBulkRow($row, $languageHeaders, $defaultLanguage);

                // Validate default language required fields have values
                if (empty($translatedFields['title'][$defaultLanguage])) {
                    return ErrorResponse(
                        labels('MULTILANGUAGE_MISSING_TITLE', "Title is required for default language ($defaultLanguage) at row " . ($rowIndex + 2)),
                        true,
                        [],
                        [],
                        200,
                        csrf_token(),
                        csrf_hash()
                    );
                }
                if (empty($translatedFields['description'][$defaultLanguage])) {
                    return ErrorResponse(
                        labels('MULTILANGUAGE_MISSING_DESCRIPTION', "Description is required for default language ($defaultLanguage) at row " . ($rowIndex + 2)),
                        true,
                        [],
                        [],
                        200,
                        csrf_token(),
                        csrf_hash()
                    );
                }
                // Process other images
                $other_images = [];
                foreach ($other_image_Headers as $indexes) {
                    $other_image = isset($row[$indexes]) ? trim($row[$indexes]) : '';
                    if (!empty($other_image) && !in_array($other_image, $old_other_images)) {
                        $oi = copy_image($row[$indexes], '/public/uploads/services/');
                        if (!empty($oi)) {
                            $other_images[] = $oi;
                        }
                    }
                }
                // If no new images, keep old ones
                if (empty($other_images) && !empty($old_other_images)) {
                    $other_images = $old_other_images;
                }

                // Process files
                $files = [];
                foreach ($FilesHeaders as $indexes) {
                    $file = isset($row[$indexes]) ? trim($row[$indexes]) : '';
                    if (!empty($file) && !in_array($file, $old_files)) {
                        $uploadedFile = copy_image($row[$indexes], '/public/uploads/services/');
                        if (!empty($uploadedFile)) {
                            $files[] = $uploadedFile;
                        }
                    }
                }
                // If no new files, keep old ones
                if (empty($files) && !empty($old_files)) {
                    $files = $old_files;
                }

                // Process main service image (FIXED: was $row[21], now $row[17])
                $image = !empty($row[17]) ? copy_image($row[17], '/public/uploads/services/') : "";
                if (empty($image) && !empty($fetch_service_data)) {
                    $image = $fetch_service_data[0]['image'];
                }

                // Prepare service data INCLUDING default language values for fallback
                // Store default language translatable fields in main table as fallback
                // These same values will also be stored in translated_service_details table
                $serviceData = [
                    'id' => $row[0], // Service ID
                    'user_id' => $row[1], // Index 1: Provider ID
                    'category_id' => $row[2], // Index 2: Category ID
                    // ✅ Include default language translatable fields as fallback
                    'title' => $translatedFields['title'][$defaultLanguage] ?? '',
                    'description' => $translatedFields['description'][$defaultLanguage] ?? '',
                    'long_description' => $translatedFields['long_description'][$defaultLanguage] ?? '',
                    'tags' => $translatedFields['tags'][$defaultLanguage] ?? '',
                    'faqs' => $translatedFields['faqs'][$defaultLanguage] ?? json_encode([], JSON_UNESCAPED_UNICODE),
                    // Non-translatable fields (FIXED: all indexes corrected)
                    'duration' => $row[3], // FIXED: was $row[6], now $row[3]
                    'number_of_members_required' => $row[4], // FIXED: was $row[7], now $row[4]
                    'max_quantity_allowed' => $row[5], // FIXED: was $row[8], now $row[5]
                    'tax_type' => $row[6], // FIXED: was $row[10], now $row[6]
                    'tax_id' => $row[7], // FIXED: was $row[11], now $row[7]
                    'price' => $row[8], // FIXED: was $row[12], now $row[8]
                    'discounted_price' => $row[9], // FIXED: was $row[13], now $row[9]
                    'is_cancelable' => $row[10], // FIXED: was $row[14], now $row[10]
                    'cancelable_till' => ($row[10] == 1) ? $row[11] : "", // FIXED: indexes corrected
                    'is_pay_later_allowed' => $row[12], // FIXED: was $row[16], now $row[12]
                    'at_store' => $row[13], // FIXED: was $row[17], now $row[13]
                    'at_doorstep' => $row[14], // FIXED: was $row[18], now $row[14]
                    'status' => $row[15], // FIXED: was $row[19], now $row[15]
                    'approved_by_admin' => ($provider[0]['need_approval_for_the_service'] == "1") ? "0" : "1",
                    'other_images' => json_encode($other_images),
                    'image' => $image,
                    'files' => json_encode($files),
                ];

                $services[] = $serviceData;
                $serviceTranslations[] = $translatedFields;
            }

            // Update services and their translations
            $serviceModel = new Service_model();
            $db = \Config\Database::connect();
            $db->transStart();

            try {
                foreach ($services as $index => $service) {
                    $serviceId = $service['id'];
                    unset($service['id']);

                    // Update service with default language values as fallback
                    if (!$serviceModel->update($serviceId, $service)) {
                        throw new \Exception("Failed to update service ID " . $serviceId . " at row " . ($index + 2));
                    }

                    // Save ALL language translations (including default) in translated_service_details table

                    // Prepare translation data for ServicesService
                    $translationData = $serviceTranslations[$index];

                    // Convert FAQ format for ServicesService if needed
                    if (isset($translationData['faqs_by_number']) && !empty($translationData['faqs_by_number'])) {
                        // ServicesService expects FAQs in the format: faqs_by_number
                        $translationData['faqs'] = $translationData['faqs_by_number'];
                        unset($translationData['faqs_by_number']);
                    }

                    $translationResult = $this->serviceService->saveTranslatedFields(
                        $serviceId,
                        $translationData
                    );

                    if (!$translationResult['success']) {
                        $errors = implode(', ', $translationResult['errors']);
                        throw new \Exception("Failed to save translations for service ID $serviceId: " . $errors);
                    }
                }

                $db->transComplete();

                if ($db->transStatus() === false) {
                    throw new \Exception('Transaction failed');
                }

                return successResponse("Services updated successfully", false, [], [], 200, csrf_token(), csrf_hash());
            } catch (\Exception $e) {
                $db->transRollback();
                log_message('error', 'Bulk update service error: ' . $e->getMessage());
                return ErrorResponse($e->getMessage(), true, [], [], 200, csrf_token(), csrf_hash());
            }
        }
    }

    public function downloadSampleForInsert()
    {
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            $_SESSION['toastMessage'] = $result['message'];
            $_SESSION['toastMessageType'] = 'error';
            $this->session->markAsFlashdata('toastMessage');
            $this->session->markAsFlashdata('toastMessageType');
            return redirect()->to('admin/services/bulk_import_services')->withCookies();
        }
        try {
            // Get available languages from database
            $languages = fetch_details('languages', [], ['code', 'language', 'is_default'], "", '0', 'id', 'ASC');

            // Build headers with non-translatable fields first
            $headers = [
                'Provider ID',
                'Category ID',
                'Duration to perform task',
                'Members Required to Perform Task',
                'Max Quantity allowed for services',
                'Price Type',
                'Tax ID',
                'Price',
                'Discounted Price',
                'Is Cancelable',
                'Cancelable before',
                'Pay Later Allowed',
                'At Store',
                'At Doorstep',
                'Status',
                'Approve Service',
                'Image',
            ];

            // Add translatable fields for each language
            // Format: Title[en], Title[es], Description[en], Description[es], etc.
            foreach ($languages as $language) {
                $langCode = $language['code'];
                $langName = $language['language'];

                // Add headers for each translatable field per language
                $headers[] = "Title[$langCode]"; // translatable
                $headers[] = "Description[$langCode]"; // translatable (short description)
                $headers[] = "Long Description[$langCode]"; // translatable
                $headers[] = "Tags[$langCode]"; // translatable

                // Add FAQ headers for this language (2 FAQs as example)
                $headers[] = "faq[$langCode][question][1]";
                $headers[] = "faq[$langCode][answer][1]";
                $headers[] = "faq[$langCode][question][2]";
                $headers[] = "faq[$langCode][answer][2]";
            }

            // Add non-translatable fields at the end
            $headers[] = 'Other Image[1]';
            $headers[] = 'Other Image[2]';
            $headers[] = 'Files[1]';
            $headers[] = 'Files[2]';

            // Build sample data row
            $sampleRow = [
                '1', // Provider ID
                '2', // Category ID
                '60', // Duration to perform task
                '1', // Members Required to Perform Task
                '5', // Max Quantity allowed for services
                'included', // Price Type
                '1', // Tax ID
                '100', // Price
                '80', // Discounted Price
                '1', // Is Cancelable
                '24', // Cancelable before (hours)
                '1', // Pay Later Allowed
                '1', // At Store
                '1', // At Doorstep
                '1', // Status (1=active)
                '1', // Approve Service
                'public/uploads/services/sample.jpg', // Image
            ];

            // Add sample translatable fields for each language
            foreach ($languages as $language) {
                $langCode = $language['code'];
                $langName = $language['language'];

                // Sample data varies by language for demonstration
                if ($langCode === 'en') {
                    $sampleRow[] = 'House Cleaning Service'; // Title[en]
                    $sampleRow[] = 'Professional house cleaning service'; // Description[en]
                    $sampleRow[] = 'We provide thorough cleaning of your home including all rooms, kitchen, and bathrooms'; // Long Description[en]
                    $sampleRow[] = 'cleaning,house,professional'; // Tags[en]
                    $sampleRow[] = 'What areas do you clean?'; // faq[en][question][1]
                    $sampleRow[] = 'We clean all rooms, kitchen, bathrooms, and common areas'; // faq[en][answer][1]
                    $sampleRow[] = 'How long does it take?'; // faq[en][question][2]
                    $sampleRow[] = 'Typically 2-3 hours depending on home size'; // faq[en][answer][2]
                } else {
                    // For other languages, leave empty (optional)
                    $sampleRow[] = 'Other language title'; // Title
                    $sampleRow[] = 'Other language description'; // Description
                    $sampleRow[] = 'Other language long description'; // Long Description
                    $sampleRow[] = 'Other language tags'; // Tags
                    $sampleRow[] = 'Other language faq[question][1]'; // faq[question][1]
                    $sampleRow[] = 'Other language faq[answer][1]'; // faq[answer][1]
                    $sampleRow[] = 'Other language faq[question][2]'; // faq[question][2]
                    $sampleRow[] = 'Other language faq[answer][2]'; // faq[answer][2]
                }
            }

            // Add sample other images and files
            $sampleRow[] = 'public/uploads/services/image1.jpg'; // Other Image[1]
            $sampleRow[] = 'public/uploads/services/image2.jpg'; // Other Image[2]
            $sampleRow[] = 'public/uploads/services/document1.pdf'; // Files[1]
            $sampleRow[] = 'public/uploads/services/document2.pdf'; // Files[2]

            $output = fopen('php://output', 'w');
            if ($output === false) {
                throw new \Exception('Failed to open output stream.');
            }
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="service_sample_without_data_multilanguage.csv"');
            fputcsv($output, $headers);
            fputcsv($output, $sampleRow); // Add sample data row
            fclose($output);
            exit;
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - download-sample-for-insert()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function downloadSampleForUpdate()
    {
        $result = checkModificationInDemoMode($this->superadmin);
        if ($result !== true) {
            $response = [
                'type' => 'error',
                'message' => $result['message']
            ];
            return $this->response->setJSON($response);
        }
        try {
            // Get active languages for multilanguage support
            $languageModel = new Language_model();
            $languages = $languageModel->findAll();
            $defaultLanguage = get_settings('default_language', true);

            // Build headers - simple string array
            $headers = [];
            $headers[] = 'ID';
            $headers[] = 'Provider ID';
            $headers[] = 'Category ID';
            $headers[] = 'Duration to perform task';
            $headers[] = 'Members Required to Perform Task';
            $headers[] = 'Max Quantity allowed for services';
            $headers[] = 'Price Type';
            $headers[] = 'Tax ID';
            $headers[] = 'Price';
            $headers[] = 'Discounted Price';
            $headers[] = 'Is Cancelable';
            $headers[] = 'Cancelable before';
            $headers[] = 'Pay Later Allowed';
            $headers[] = 'At Store';
            $headers[] = 'At Doorstep';
            $headers[] = 'Status';
            $headers[] = 'Approve Service';
            $headers[] = 'Image';

            // Add language-specific headers
            foreach ($languages as $language) {
                $langCode = $language['code'];
                $headers[] = "Title[$langCode]";
                $headers[] = "Description[$langCode]";
                $headers[] = "Long Description[$langCode]";
                $headers[] = "Tags[$langCode]";
                $headers[] = "faq[$langCode][question][1]";
                $headers[] = "faq[$langCode][answer][1]";
                $headers[] = "faq[$langCode][question][2]";
                $headers[] = "faq[$langCode][answer][2]";
            }

            $headers[] = 'Other Image[1]';
            $headers[] = 'Other Image[2]';
            $headers[] = 'Files[1]';
            $headers[] = 'Files[2]';

            // Fetch services
            $partners = $this->request->getPost('partners');
            if (!empty($partners)) {
                $services = fetch_details('services', [], [], "", 0, 'id', 'DESC', 'user_id', $partners);
            } else {
                $services = fetch_details('services');
            }

            // Prepare data
            $all_data = [];
            $translationModel = new TranslatedServiceDetails_model();

            foreach ($services as $service) {
                $row = [];

                // log_message('debug', '=== Processing Service ID: ' . ($service['id'] ?? 'UNKNOWN') . ' ===');

                // Add basic service fields - force to string immediately
                $row[] = strval($service['id'] ?? '');
                $row[] = strval($service['user_id'] ?? '');
                $row[] = strval($service['category_id'] ?? '');
                $row[] = strval($service['duration'] ?? '');
                $row[] = strval($service['number_of_members_required'] ?? '');
                $row[] = strval($service['max_quantity_allowed'] ?? '');
                $row[] = strval($service['tax_type'] ?? '');
                $row[] = strval($service['tax_id'] ?? '');
                $row[] = strval($service['price'] ?? '');
                $row[] = strval($service['discounted_price'] ?? '');
                $row[] = strval($service['is_cancelable'] ?? '');
                $row[] = strval($service['cancelable_till'] ?? '');
                $row[] = strval($service['is_pay_later_allowed'] ?? '');
                $row[] = strval($service['at_store'] ?? '');
                $row[] = strval($service['at_doorstep'] ?? '');
                $row[] = strval($service['status'] ?? '');
                $row[] = strval($service['approved_by_admin'] ?? '');
                $row[] = strval($service['image'] ?? '');

                // log_message('debug', 'Basic fields added. Row count: ' . count($row));

                // Get translations
                $translations = $translationModel->where('service_id', $service['id'])->findAll();
                // log_message('debug', 'Found ' . count($translations) . ' translations');

                $translationsByLang = [];
                foreach ($translations as $trans) {
                    $translationsByLang[$trans['language_code']] = $trans;
                    // log_message('debug', 'Translation for language: ' . $trans['language_code']);
                }

                // Add language-specific data
                foreach ($languages as $language) {
                    $langCode = $language['code'];
                    // log_message('debug', 'Processing language: ' . $langCode);

                    if (isset($translationsByLang[$langCode])) {
                        $trans = $translationsByLang[$langCode];

                        // log_message('debug', 'Translation data types - title: ' . gettype($trans['title'] ?? null) . ', tags: ' . gettype($trans['tags'] ?? null));

                        // Add translatable text fields
                        $row[] = strval($trans['title'] ?? '');
                        $row[] = strval($trans['description'] ?? '');
                        $row[] = strval(strip_tags(htmlspecialchars_decode(stripslashes($trans['long_description'] ?? ''))));
                        $row[] = strval($trans['tags'] ?? '');

                        // Handle FAQs
                        $faqsJson = $trans['faqs'] ?? '[]';
                        // log_message('debug', 'FAQs JSON length: ' . strlen($faqsJson) . ', First 100 chars: ' . substr($faqsJson, 0, 100));

                        $faqs = @json_decode($faqsJson, true);
                        // log_message('debug', 'FAQs decoded - type: ' . gettype($faqs) . ', count: ' . (is_array($faqs) ? count($faqs) : 0));

                        // if (!is_array($faqs)) {
                        //     log_message('debug', 'FAQs is not array, resetting to empty');
                        //     $faqs = [];
                        // }

                        // Add first FAQ
                        if (isset($faqs[0]) && is_array($faqs[0])) {
                            // log_message('debug', 'FAQ[0] keys: ' . implode(', ', array_keys($faqs[0])));
                            // log_message('debug', 'FAQ[0] question type: ' . gettype($faqs[0]['question'] ?? null));
                            $row[] = strval($faqs[0]['question'] ?? '');
                            $row[] = strval($faqs[0]['answer'] ?? '');
                        } else {
                            // log_message('debug', 'FAQ[0] not found or not array');
                            $row[] = '';
                            $row[] = '';
                        }

                        // Add second FAQ
                        if (isset($faqs[1]) && is_array($faqs[1])) {
                            $row[] = strval($faqs[1]['question'] ?? '');
                            $row[] = strval($faqs[1]['answer'] ?? '');
                        } else {
                            $row[] = '';
                            $row[] = '';
                        }
                    } else {
                        // log_message('debug', 'No translation for language: ' . $langCode);
                        // No translation for this language - add 8 empty strings
                        $row[] = '';
                        $row[] = '';
                        $row[] = '';
                        $row[] = '';
                        $row[] = '';
                        $row[] = '';
                        $row[] = '';
                        $row[] = '';
                    }
                }

                // Handle other_images
                $otherImagesJson = $service['other_images'] ?? '[]';
                // log_message('debug', 'Other images JSON: ' . substr($otherImagesJson, 0, 100));
                $otherImages = @json_decode($otherImagesJson, true);
                // log_message('debug', 'Other images decoded - type: ' . gettype($otherImages) . ', count: ' . (is_array($otherImages) ? count($otherImages) : 0));

                if (!is_array($otherImages)) {
                    $otherImages = [];
                }

                // if (isset($otherImages[0])) {
                //     log_message('debug', 'OtherImages[0] type: ' . gettype($otherImages[0]));
                // }

                $row[] = isset($otherImages[0]) && is_string($otherImages[0]) ? $otherImages[0] : '';
                $row[] = isset($otherImages[1]) && is_string($otherImages[1]) ? $otherImages[1] : '';

                // Handle files
                $filesJson = $service['files'] ?? '[]';
                $files = @json_decode($filesJson, true);
                if (!is_array($files)) {
                    $files = [];
                }
                $row[] = isset($files[0]) && is_string($files[0]) ? $files[0] : '';
                $row[] = isset($files[1]) && is_string($files[1]) ? $files[1] : '';

                // Log final row to check for any non-string values
                // log_message('debug', 'Final row count: ' . count($row));
                foreach ($row as $index => $value) {
                    $type = gettype($value);
                    if ($type !== 'string') {
                        log_message('error', 'NON-STRING VALUE at index ' . $index . ' - Type: ' . $type . ', Value: ' . print_r($value, true));
                    }
                }

                $all_data[] = $row;
            }

            // log_message('debug', 'Total services processed: ' . count($all_data));

            // Output CSV
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename="service_update_sample_with_data_multilanguage.csv"');

            $output = fopen('php://output', 'w');
            fputcsv($output, $headers);

            foreach ($all_data as $row) {
                fputcsv($output, $row);
            }

            fclose($output);
            exit;
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - downloadSampleForUpdate()');
            header('Content-Type: application/json');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function ServiceAddInstructions()
    {
        try {
            $filePath = (FCPATH . '/public/uploads/site/Service-Add-Instructions.pdf');
            $fileName = 'Service-Add-Instructions.pdf';
            if (file_exists($filePath)) {
                return $this->response->download($filePath, null)->setFileName($fileName);
            } else {
                $_SESSION['toastMessage'] = labels('cant_download', 'Cannot download');
                $_SESSION['toastMessageType'] = 'error';
                $this->session->markAsFlashdata('toastMessage');
                $this->session->markAsFlashdata('toastMessageType');
                return redirect()->to('admin/services/bulk_import_services')->withCookies();
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - download_service_add_instruction_file()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    public function ServiceUpdateInstructions()
    {
        try {
            $filePath = (FCPATH . '/public/uploads/site/Service-Update-Instructions.pdf');
            $fileName = 'Service-Update-Instructions.pdf';
            if (file_exists($filePath)) {
                return $this->response->download($filePath, null)->setFileName($fileName);
            } else {
                $_SESSION['toastMessage'] = labels('cant_download', 'Cannot download');
                $_SESSION['toastMessageType'] = 'error';
                $this->session->markAsFlashdata('toastMessage');
                $this->session->markAsFlashdata('toastMessageType');
                return redirect()->to('admin/services')->withCookies();
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - download_service_add_instruction_file()');
            return ErrorResponse(labels(SOMETHING_WENT_WRONG, "Something Went Wrong"), true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    /**
     * Remove SEO image for a service
     * This method handles AJAX requests to remove SEO images
     * @return \CodeIgniter\HTTP\Response
     */
    public function remove_seo_image()
    {
        try {
            if (!$this->isLoggedIn || !$this->userIsAdmin) {
                return ErrorResponse("Unauthorized access", true, [], [], 200, csrf_token(), csrf_hash());
            }

            $serviceId = $this->request->getPost('service_id');
            $seoId = $this->request->getPost('seo_id');

            if (!$serviceId) {
                return ErrorResponse("Service ID is required", true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Set SEO model context for services
            $this->seoModel->setTableContext('services');

            // Get existing SEO settings
            $existingSettings = $this->seoModel->getSeoSettingsByReferenceId($serviceId);

            if (!$existingSettings) {
                return ErrorResponse("SEO settings not found for this service", true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Check if there's an image to remove
            if (empty($existingSettings['image'])) {
                return ErrorResponse("No SEO image found to remove", true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Store the image name for cleanup
            $imageToDelete = $existingSettings['image'];

            // Prepare update data - remove image but keep other fields
            $updateData = [
                'title' => $existingSettings['title'] ?? '',
                'description' => $existingSettings['description'] ?? '',
                'keywords' => $existingSettings['keywords'] ?? '',
                'schema_markup' => $existingSettings['schema_markup'] ?? '',
                'image' => '', // Clear the image field
                'service_id' => $serviceId
            ];

            // Check if all other SEO fields are empty
            $hasOtherSeoData = !empty($updateData['title']) ||
                !empty($updateData['description']) ||
                !empty($updateData['keywords']) ||
                !empty($updateData['schema_markup']);

            // If all other fields are empty, we should NOT delete the record
            // Instead, we keep the record with empty image but preserve the structure
            // This ensures the SEO record exists for future use
            $result = $this->seoModel->updateSeoSettings($existingSettings['id'], $updateData);

            if (!empty($result['error'])) {
                return ErrorResponse($result['message'], true, [], [], 200, csrf_token(), csrf_hash());
            }

            // Clean up the image file from storage
            if (!empty($imageToDelete)) {
                $disk = fetch_current_file_manager();
                delete_file_based_on_server('service_seo_settings', $imageToDelete, $disk);
            }

            return successResponse("SEO image removed successfully", false, [], [], 200, csrf_token(), csrf_hash());
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - remove_seo_image()');
            return ErrorResponse("Something went wrong while removing SEO image", true, [], [], 200, csrf_token(), csrf_hash());
        }
    }

    private function saveServiceSeoSettings(int $serviceId): void
    {
        try {
            // Parse meta keywords (Tagify or comma-separated)
            $metaKeywords = $this->request->getPost('meta_keywords');
            $keywords = $this->parseKeywords($metaKeywords);

            // Build SEO data array
            $seoData = [
                'title'         => trim((string) $this->request->getPost('meta_title')),
                'description'   => trim((string) $this->request->getPost('meta_description')),
                'keywords'      => $keywords,
                'schema_markup' => trim((string) $this->request->getPost('schema_markup')),
                'service_id'    => $serviceId,
            ];

            // Check if any SEO field is filled (excluding service_id)
            $hasSeoData = array_filter($seoData, fn($v) => !empty($v) && $v !== $serviceId);

            // Check if all SEO fields are intentionally cleared
            $allFieldsCleared = empty($seoData['title']) &&
                empty($seoData['description']) &&
                empty($seoData['keywords']) &&
                empty($seoData['schema_markup']);

            // Handle SEO image upload
            $seoImage = $this->request->getFile('meta_image');
            $hasImage = $seoImage && $seoImage->isValid();

            // Use Seo_model for service context
            $this->seoModel->setTableContext('services');
            $existingSettings = $this->seoModel->getSeoSettingsByReferenceId($serviceId);

            $newSeoData = $seoData;
            if ($hasImage) {
                $uploadResult = upload_file(
                    $seoImage,
                    'public/uploads/seo_settings/service_seo_settings/',
                    'Failed to upload SEO image for service',
                    'service_seo_settings'
                );
                if ($uploadResult['error']) {
                    throw new Exception('SEO image upload failed: ' . $uploadResult['message']);
                }
                $newSeoData['image'] = $uploadResult['file_name'];
            } else {
                $newSeoData['image'] = $existingSettings['image'] ?? '';
            }

            // If no existing settings, create new if data or image exists
            if (!$existingSettings) {
                if ($hasSeoData || $hasImage) {
                    $result = $this->seoModel->createSeoSettings($newSeoData);
                    if (!empty($result['error'])) {
                        $errors = $result['validation_errors'] ?? [];
                        throw new Exception($result['message'] . (!empty($errors) ? ': ' . json_encode($errors) : ''));
                    }
                }
                return;
            }

            // If existing settings exist and all fields are cleared (and no new image), delete the record
            // BUT: If there's an existing image, we should NOT delete the record even if all other fields are empty
            // This preserves the SEO record structure for future use
            if ($existingSettings && $allFieldsCleared && !$hasImage && empty($existingSettings['image'])) {
                $result = $this->seoModel->delete($existingSettings['id']);
                if ($result) {
                    // Clean up old image if it exists
                    if (!empty($existingSettings['image'])) {
                        $disk = fetch_current_file_manager();
                        delete_file_based_on_server('service_seo_settings', $existingSettings['image'], $disk);
                    }
                }
                return;
            }

            // Compare existing and new settings
            $settingsChanged = false;
            foreach ($newSeoData as $key => $value) {
                $existingValue = $existingSettings[$key] ?? '';
                $newValue = $value ?? '';
                if ($existingValue !== $newValue) {
                    $settingsChanged = true;
                    break;
                }
            }

            // Also check if a new image was uploaded (this forces an update)
            if (!$settingsChanged && $hasImage) {
                $settingsChanged = true;
            }

            if (!$settingsChanged) {
                return;
            }

            // Update existing settings with new data
            $result = $this->seoModel->updateSeoSettings($existingSettings['id'], $newSeoData);
            if (!empty($result['error'])) {
                $errors = $result['validation_errors'] ?? [];
                throw new Exception($result['message'] . (!empty($errors) ? ': ' . json_encode($errors) : ''));
            }
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/admin/Services.php - saveServiceSeoSettings()');
            throw $th; // Re-throw to handle in add_service
        }
    }

    private function parseKeywords($input): string
    {
        // If input is empty, return empty string
        if (empty($input)) {
            return '';
        }

        // If input is a string, it might be JSON or comma-separated
        if (is_string($input)) {
            // Check if it's a JSON string
            if (json_decode($input, true) !== null) {
                $decoded = json_decode($input, true);
                if (is_array($decoded)) {
                    // Handle array of objects (e.g., [{value: "tag1"}, {value: "tag2"}])
                    $tags = array_map(function ($item) {
                        return is_array($item) && isset($item['value']) ? trim($item['value']) : trim($item);
                    }, $decoded);
                    return implode(',', $tags);
                }
            }
            // Treat as comma-separated string
            return trim($input);
        }

        // If input is an array
        if (is_array($input)) {
            // Handle case where array contains a single JSON string (e.g., ['[{value: "tag1"}, {value: "tag2"}]'])
            if (count($input) === 1 && is_string($input[0]) && json_decode($input[0], true) !== null) {
                $decoded = json_decode($input[0], true);
                if (is_array($decoded)) {
                    $tags = array_map(function ($item) {
                        return is_array($item) && isset($item['value']) ? trim($item['value']) : trim($item);
                    }, $decoded);
                    return implode(',', $tags);
                }
            }
            // Handle array of objects (e.g., [{value: "tag1"}, {value: "tag2"}])
            $tags = array_map(function ($item) {
                return is_array($item) && isset($item['value']) ? trim($item['value']) : trim($item);
            }, $input);
            return implode(',', $tags);
        }

        // Fallback: return empty string for unexpected input
        return '';
    }

    /**
     * Process tags value and convert to comma-separated string
     * 
     * @param mixed $tagsValue The tags value from form data
     * @return string Comma-separated string of tag values
     */
    private function processTagsValue($tagsValue): string
    {
        if (empty($tagsValue)) {
            return '';
        }

        // If it's already a string, return it trimmed
        if (is_string($tagsValue)) {
            return trim($tagsValue);
        }

        // If it's an array, process it
        if (is_array($tagsValue)) {
            $tagValues = [];

            foreach ($tagsValue as $tag) {
                if (is_string($tag)) {
                    // Check if it's a JSON string
                    $decoded = json_decode($tag, true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                        // It's a JSON string, process the decoded array
                        foreach ($decoded as $item) {
                            if (is_array($item) && isset($item['value'])) {
                                $tagValues[] = trim($item['value']);
                            }
                        }
                    } else {
                        // Direct string value
                        $tagValues[] = trim($tag);
                    }
                } elseif (is_array($tag)) {
                    // Check if it's a direct object format like {"value":"Test"}
                    if (isset($tag['value'])) {
                        $tagValues[] = trim($tag['value']);
                    } else {
                        // Handle case where $tag is an array like [{"value":"Test"}]
                        foreach ($tag as $tagItem) {
                            if (is_array($tagItem) && isset($tagItem['value'])) {
                                $tagValues[] = trim($tagItem['value']);
                            } elseif (is_string($tagItem)) {
                                $tagValues[] = trim($tagItem);
                            }
                        }
                    }
                }
            }

            // Remove empty values and return as comma-separated string
            $filteredValues = array_filter($tagValues, function ($value) {
                return !empty(trim($value));
            });

            $result = implode(', ', $filteredValues);
            return $result;
        }

        // Fallback: convert to string
        return trim((string)$tagsValue);
    }

    /**
     * Process clean, language-grouped FAQ data structure
     * 
     * @param array $faqData The FAQ data in clean format: {"en": [["q1", "a1"], ["q2", "a2"]], "hi": [["q1", "a1"]]}
     * @return array Processed FAQ data for translation fields
     */
    private function processCleanFAQData(array $faqData): array
    {
        $translatedFields = [];
        $defaultLanguage = get_default_language();

        // Process FAQs for each language
        foreach ($faqData as $languageCode => $languageFaqs) {
            // Skip invalid language codes
            if (empty($languageCode) || !is_array($languageFaqs)) {
                continue;
            }

            $processedFaqs = [];

            // Process each FAQ pair in the language
            foreach ($languageFaqs as $faqPair) {
                // Skip if FAQ pair is not an array or doesn't have exactly 2 elements
                if (!is_array($faqPair) || count($faqPair) !== 2) {
                    continue;
                }

                $question = trim($faqPair[0] ?? '');
                $answer = trim($faqPair[1] ?? '');

                // Only add FAQ if either question or answer is not empty
                if (!empty($question) || !empty($answer)) {
                    $processedFaqs[] = [
                        'question' => $question,
                        'answer' => $answer
                    ];
                }
            }

            // Store processed FAQs for this language
            if (!empty($processedFaqs)) {
                $translatedFields['faqs'][$languageCode] = $processedFaqs; // Store as array, not JSON string
            } else {
                // Store empty array for languages with no FAQs
                $translatedFields['faqs'][$languageCode] = []; // Store as array, not JSON string
            }
        }

        // Ensure all languages have FAQ entries (even if empty)
        $languages = fetch_details('languages', [], ['code']);
        foreach ($languages as $language) {
            $languageCode = $language['code'];
            if (!isset($translatedFields['faqs'][$languageCode])) {
                $translatedFields['faqs'][$languageCode] = json_encode([], JSON_UNESCAPED_UNICODE);
            }
        }

        // log_message('debug', 'Admin - Processed clean FAQ data: ' . json_encode($translatedFields['faqs']));

        return $translatedFields;
    }

    /**
     * Transform form data to translated fields structure with clean FAQ handling
     * 
     * @param array $postData The form post data
     * @param string $defaultLanguage The default language code
     * @param int|null $serviceId The service ID for updates
     * @param array $existingTranslations Existing translations data
     * @return array Transformed fields for translation
     */
    private function transformFormDataToTranslatedFields(array $postData, string $defaultLanguage, ?int $serviceId = null, array $existingTranslations = []): array
    {
        // LOG: Start of transformation
        // log_message('debug', '[TRANSLATION_TRANSFORM] Starting transformation for service ID: ' . ($serviceId ?? 'new'));
        // log_message('debug', '[TRANSLATION_TRANSFORM] Default language: ' . $defaultLanguage);
        // log_message('debug', '[TRANSLATION_TRANSFORM] Has existing translations: ' . (empty($existingTranslations) ? 'No' : 'Yes'));

        $translatedFields = [];
        $translatableFields = ['title', 'description', 'long_description', 'tags', 'faqs'];

        // Check if we have clean FAQ data from the new structure
        if (isset($postData['faqs']) && is_string($postData['faqs'])) {
            $faqData = json_decode($postData['faqs'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($faqData)) {
                // Process clean FAQ data
                $cleanFaqFields = $this->processCleanFAQData($faqData);
                $translatedFields = array_merge($translatedFields, $cleanFaqFields);

                // Remove the raw FAQ data from postData to avoid double processing
                unset($postData['faqs']);
            }
        } else {
            // Check if FAQs are in array format
            if (isset($postData['faqs']) && is_array($postData['faqs'])) {
                // Process FAQs from array format
                foreach ($postData['faqs'] as $languageCode => $languageFaqs) {
                    if (is_array($languageFaqs)) {
                        $translatedFields['faqs'][$languageCode] = $languageFaqs;
                    }
                }
            }
        }

        // Process other translatable fields
        foreach ($translatableFields as $field) {
            // Skip FAQs as they're already processed above
            if ($field === 'faqs') {
                continue;
            }

            // log_message('debug', '[TRANSLATION_TRANSFORM] Processing field: ' . $field);

            if (isset($postData[$field]) && is_array($postData[$field])) {
                // log_message('debug', '[TRANSLATION_TRANSFORM] Field ' . $field . ' data: ' . json_encode($postData[$field], JSON_UNESCAPED_UNICODE));

                foreach ($postData[$field] as $languageCode => $value) {
                    // Skip invalid language codes
                    if (empty($languageCode) || $languageCode === '0') {
                        // log_message('debug', '[TRANSLATION_TRANSFORM] Skipping invalid language code: ' . $languageCode);
                        continue;
                    }

                    if ($field === 'tags') {
                        // Process tags to comma-separated string
                        $processedValue = $this->processTagsValue($value);
                        $translatedFields[$field][$languageCode] = $processedValue;
                        // log_message('debug', '[TRANSLATION_TRANSFORM] Processed tags for ' . $languageCode . ': ' . $processedValue);
                    } else {
                        // For other fields, just trim the value
                        $processedValue = trim($value);
                        $translatedFields[$field][$languageCode] = $processedValue;
                        // log_message('debug', '[TRANSLATION_TRANSFORM] Processed ' . $field . ' for ' . $languageCode . ': ' . $processedValue);
                    }
                }
            } else {
                // log_message('debug', '[TRANSLATION_TRANSFORM] Field ' . $field . ' not found in POST data or not array. Checking existing translations...');

                // If field is not in form data but this is an update, preserve existing data
                if ($serviceId && isset($existingTranslations)) {
                    foreach ($existingTranslations as $languageCode => $translation) {
                        if (isset($translation[$field]) && !empty($translation[$field])) {
                            $translatedFields[$field][$languageCode] = $translation[$field];
                            // log_message('debug', '[TRANSLATION_TRANSFORM] Preserved existing ' . $field . ' for ' . $languageCode . ': ' . $translation[$field]);
                        }
                    }
                } else {
                    // log_message('debug', '[TRANSLATION_TRANSFORM] No existing translations to preserve for field: ' . $field);
                }
            }
        }

        // log_message('debug', '[TRANSLATION_TRANSFORM] Final transformed fields: ' . json_encode($translatedFields, JSON_UNESCAPED_UNICODE));
        return $translatedFields;
    }

    /**
     * Extract translated fields from bulk CSV row data
     * 
     * This method extracts translatable fields (title, description, long_description, tags, faqs)
     * from a CSV row and organizes them by language code for bulk import operations.
     * 
     * @param array $row CSV row data
     * @param array $languageHeaders Parsed language header mappings from CSV
     * @param string $defaultLanguage Default language code
     * @return array Translated fields organized by field name and language code
     */
    private function extractTranslatedFieldsFromBulkRow(array $row, array $languageHeaders, string $defaultLanguage): array
    {
        // Initialize structure for all translatable fields
        $translatedFields = [
            'title' => [],
            'description' => [],
            'long_description' => [],
            'tags' => [],
            'faqs' => []
        ];

        // Extract simple translatable fields: title, description, long_description, tags
        foreach (['title', 'description', 'long_description', 'tags'] as $field) {
            if (isset($languageHeaders[$field])) {
                foreach ($languageHeaders[$field] as $langCode => $columnIndex) {
                    $value = isset($row[$columnIndex]) ? trim($row[$columnIndex]) : '';
                    if (!empty($value)) {
                        $translatedFields[$field][$langCode] = $value;
                    }
                }
            }
        }

        // Extract FAQs (more complex structure)
        // FAQs are stored as: faq[en][question][1], faq[en][answer][1], etc.
        // We need to process FAQs for both main table storage and translations table
        if (isset($languageHeaders['faqs'])) {
            // Group FAQs by language first for main table storage
            $faqsByLanguage = [];
            // Also group by FAQ number for ServicesService format
            $faqsByNumber = [];

            foreach ($languageHeaders['faqs'] as $langCode => $faqNumbers) {
                $languageFaqs = [];

                foreach ($faqNumbers as $faqNumber => $questionAnswer) {
                    $question = isset($questionAnswer['question']) && isset($row[$questionAnswer['question']])
                        ? trim($row[$questionAnswer['question']])
                        : '';
                    $answer = isset($questionAnswer['answer']) && isset($row[$questionAnswer['answer']])
                        ? trim($row[$questionAnswer['answer']])
                        : '';

                    // Only add FAQ if question or answer is not empty
                    if (!empty($question) || !empty($answer)) {
                        $faqData = [
                            'question' => $question,
                            'answer' => $answer
                        ];

                        // Add to language-based structure for main table
                        $languageFaqs[] = $faqData;

                        // Add to FAQ number-based structure for ServicesService
                        if (!isset($faqsByNumber[$faqNumber])) {
                            $faqsByNumber[$faqNumber] = [];
                        }
                        $faqsByNumber[$faqNumber][$langCode] = $faqData;
                    }
                }

                // Store FAQs for this language (for main table access)
                if (!empty($languageFaqs)) {
                    $faqsByLanguage[$langCode] = json_encode($languageFaqs, JSON_UNESCAPED_UNICODE);
                } else {
                    $faqsByLanguage[$langCode] = json_encode([], JSON_UNESCAPED_UNICODE);
                }
            }

            // Store both formats
            $translatedFields['faqs'] = $faqsByLanguage; // For main table access
            $translatedFields['faqs_by_number'] = $faqsByNumber; // For ServicesService
        }

        return $translatedFields;
    }
}
