<?php

namespace App\Controllers\partner;

use App\Controllers\partner\Partner;
use App\Models\ReasonsForReportAndBlockChat_model;

class ReportController extends Partner
{
    protected $reasonsForReportAndBlockChat;

    public function __construct()
    {
        parent::__construct();
        $this->reasonsForReportAndBlockChat = new ReasonsForReportAndBlockChat_model();
    }

    public function get_report_reasons()
    {
        try {
            if (!$this->isLoggedIn) {
                return $this->response->setJSON(['error' => true, 'message' => labels(UNAUTHORIZED_ACCESS, "Unauthorized access")]);
            }

            // Get current language for translations
            $session = session();
            $currentLanguage = $session->get('lang') ?? 'en';

            // If no language set in session, try to get from database
            if ($currentLanguage === 'en') {
                $defaultLanguage = fetch_details('languages', ['is_default' => 1], ['code']);
                if (!empty($defaultLanguage)) {
                    $currentLanguage = $defaultLanguage[0]['code'];
                }
            }

            // Get default language code
            $defaultLanguage = fetch_details('languages', ['is_default' => 1], ['code'])[0]['code'] ?? 'en';

            // Get all reasons from main table with reason field for fallback
            $reasons = fetch_details('reasons_for_report_and_block_chat', ['type' => 'admin'], ['id', 'reason', 'needs_additional_info', 'type']);

            // Get translations for all reasons
            $reasonIds = array_column($reasons, 'id');
            $translatedReasonModel = new \App\Models\TranslatedReasonsForReportAndBlockChat_model();

            // Get current language translations
            $currentTranslations = [];
            if (!empty($reasonIds)) {
                $currentTranslations = $translatedReasonModel->getTranslationsForReasons($reasonIds, $currentLanguage);
            }

            // Get default language translations
            $defaultTranslations = [];
            if (!empty($reasonIds) && $currentLanguage !== $defaultLanguage) {
                $defaultTranslations = $translatedReasonModel->getTranslationsForReasons($reasonIds, $defaultLanguage);
            }

            // Create lookup arrays for translations
            $currentTranslationLookup = [];
            foreach ($currentTranslations as $translation) {
                $currentTranslationLookup[$translation['reason_id']] = $translation['reason'];
            }

            $defaultTranslationLookup = [];
            foreach ($defaultTranslations as $translation) {
                $defaultTranslationLookup[$translation['reason_id']] = $translation['reason'];
            }

            // Add translated reason text to each reason with proper fallback logic
            foreach ($reasons as &$reason) {
                // Priority 1: Current language translation
                if (isset($currentTranslationLookup[$reason['id']]) && !empty($currentTranslationLookup[$reason['id']])) {
                    $reason['reason'] = $currentTranslationLookup[$reason['id']];
                }
                // Priority 2: Default language translation
                elseif (isset($defaultTranslationLookup[$reason['id']]) && !empty($defaultTranslationLookup[$reason['id']])) {
                    $reason['reason'] = $defaultTranslationLookup[$reason['id']];
                }
                // Priority 3: Main table data (fallback)
                else {
                    $reason['reason'] = $reason['reason'] ?? '';
                }
            }

            return $this->response->setJSON([
                'error' => false,
                'message' => labels(REASONS_FETCHED_SUCCESSFULLY, "Reasons fetched successfully"),
                'data' => $reasons
            ]);
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/partner/ReportController.php - get_report_reasons()');
            return $this->response->setJSON(['error' => true, 'message' => labels(SOMETHING_WENT_WRONG, "Something went wrong")]);
        }
    }

    public function submit_report()
    {
        try {
            if (!$this->isLoggedIn) {
                return $this->response->setJSON(['error' => true, 'message' => labels(UNAUTHORIZED_ACCESS, "Unauthorized access")]);
            }

            $reported_user_id = $this->request->getPost('reported_user_id');
            $reason_id = $this->request->getPost('reason_id');
            $additional_info = $this->request->getPost('additional_info');
            $order_id = $this->request->getPost('order_id');

            // Get reason details to check if additional info is required

            if (!empty($reason_id)) {
                $reason = fetch_details('reasons_for_report_and_block_chat', ['id' => $reason_id], ['id', 'needs_additional_info', 'type'])[0];
                if ($reason['needs_additional_info'] == 1 && empty($additional_info)) {
                    return $this->response->setJSON(['error' => true, 'message' => labels(ADDITIONAL_INFORMATION_IS_REQUIRED_FOR_THIS_REASON, "Additional information is required for this reason")]);
                }
            }
            // Save the report
            $data = [
                'reporter_id' => $this->userId,
                'reported_user_id' => $reported_user_id,
                'reason_id' => $reason_id ?? null,
                'additional_info' => $additional_info,
                'created_at' => date('Y-m-d H:i:s')
            ];

            $db = \Config\Database::connect();
            $builder = $db->table('user_reports');
            $builder->insert($data);

            return $this->response->setJSON([
                'error' => false,
                'message' => labels(USER_REPORTED_AND_BLOCKED_SUCCESSFULLY, "User Reported and Blocked Successfully")
            ]);
        } catch (\Throwable $th) {
            log_the_responce($th, date("Y-m-d H:i:s") . '--> app/Controllers/partner/ReportController.php - submit_report()');
            return $this->response->setJSON(['error' => true, 'message' => labels(SOMETHING_WENT_WRONG, "Something went wrong")]);
        }
    }
}
